<?php
// cron/import_leads.php
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../config.php';
require_once GOOGLE_API_PATH . 'sheets_client.php';

// 1) Connect to DB
$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_errno) {
    error_log("тЭМ DB connect error: " . $db->connect_error);
    exit;
}

/* -----------------------------------------------------------------------------
   Header canonicalization helpers
   - Order-independent header mapping
   - Alias and substring-based fuzzy matching
----------------------------------------------------------------------------- */

// keep spaces for substring checks
function _norm_header(string $h): string {
    $h = strtolower(trim($h));
    $h = preg_replace('/\s+/', ' ', $h);          // collapse spaces
    return preg_replace('/[^a-z0-9 ]/', '', $h);  // keep letters, digits, space
}

function _snake(string $h): string {
    $h = strtolower(trim($h));
    $h = preg_replace('/[^a-z0-9]+/', '_', $h);   // non-alnum -> underscore
    return trim($h, '_');
}

// exact alias lookup (normalized variant => canonical key)
function _alias_lookup(): array {
    $aliases = [
        'id'            => ['id','lead id','lead_id','leadid'],
        'created time'  => ['created time','created_time','createdtime','timestamp','created at','createdat'],
        'phone'         => ['phone','phone number','phone_number','contact','contact number','mobile'],
        'email'         => ['email','e-mail','mail'],
        'name'          => ['name','full name','fullname','customer name','customer'],
        'country'       => ['country','nation'],
        'platform'      => ['platform','source','leadsource','lead source'],
        // Business fields with common variations
        'budget'        => ['budget','price budget','price','amount','investment','cost','budget range','price range','what is your total budget'],
        'buy_timeline'  => [
            'how soon are you planning to buy',
            'how soon are you planning to invest',
            'buying timeline',
            'purchase plan',
            'when are you planning to buy',
            'expected purchase'
        ],
    ];
    $lk = [];
    foreach ($aliases as $canonical => $vars) {
        foreach ($vars as $v) {
            $lk[_norm_header($v)] = $canonical;
        }
    }
    return $lk;
}

// substring keyword rules (last-resort after aliases)
function _keyword_rules(): array {
    return [
        'budget'       => ['budget','investment','price','amount','cost','range','total budget'],
        'buy_timeline' => ['how soon','timeline','purchase plan','buying','invest','when are you','expected'],
    ];
}

// build index -> canonical key map
function _build_header_map(array $raw): array {
    $alias = _alias_lookup();
    $rules = _keyword_rules();
    $map = [];

    foreach ($raw as $i => $h) {
        $norm = _norm_header((string)$h);

        // 1) exact alias
        if (isset($alias[$norm])) { $map[$i] = $alias[$norm]; continue; }

        // 2) substring keywords
        $assigned = null;
        foreach ($rules as $canonical => $kws) {
            foreach ($kws as $kw) {
                if (strpos($norm, $kw) !== false) { $assigned = $canonical; break 2; }
            }
        }

        // 3) fallback to snake_case of original header
        $map[$i] = $assigned ?: _snake((string)$h);
    }

    error_log('IMPORT: header_map=' . json_encode($map));
    return $map;
}

// --- Defaults helper (platform/country/name/phone mirroring) ---
function _apply_defaults(array $r, string $tabName): array {
    // mirror to API field names if only alias was captured
    if (!isset($r['full_name']) && isset($r['name'])) {
        $r['full_name'] = $r['name'];
    }
    if (!isset($r['phone_number']) && isset($r['phone'])) {
        $r['phone_number'] = $r['phone'];
    }

    // defaultable fields used in filters/search
    $plat = strtolower(trim((string)($r['platform'] ?? '')));
    if ($plat === '' || $plat === 'platform') {
        $r['platform'] = 'unknown';       // normalized for filters
    }
    $country = trim((string)($r['country'] ?? ''));
    if ($country === '' || strtolower($country) === 'country') {
        $r['country'] = 'Unknown';
    }
    // optional defaults for common business fields
    if (empty($r['buy_timeline'])) $r['buy_timeline'] = 'Unknown';
    if (empty($r['budget']))       $r['budget']       = 'Unknown';

    // stash the source tab for downstream debugging
    if (!isset($r['tab_name'])) $r['tab_name'] = $tabName;

    return $r;
}

// --- Deterministic ID when sheet "id" cell is empty ---
function _auto_lead_id(array $r, string $tabName): ?string {
    $email   = trim((string)($r['email'] ?? ''));
    $phone   = trim((string)($r['phone'] ?? $r['phone_number'] ?? ''));
    $created = trim((string)($r['created_time'] ?? ''));
    $row     = (string)($r['_rowNumber'] ?? '');
    $seed    = $email . '|' . $phone . '|' . $created . '|' . $tabName . '|' . $row;

    // If all components are empty, donтАЩt fabricate an ID
    if ($seed === '||||') return null;

    // Stable, short-ish ID; prefix "A:" to distinguish from sheet IDs
    return 'A:' . substr(sha1($seed), 0, 24);
}


/* -----------------------------------------------------------------------------
   Deterministic lead_id generator (used when sheet ID cell is empty)
   - Stable across reimports
   - Uses email / phone / created_time / tab / row number
----------------------------------------------------------------------------- */
function _first_nonempty(array $row, array $keys): string {
    foreach ($keys as $k) {
        if (!isset($row[$k])) continue;
        $v = trim((string)$row[$k]);
        if ($v !== '') return $v;
    }
    return '';
}
function _normalize_phone(string $raw): string {
    if ($raw === '') return '';
    $digits = preg_replace('/\D+/', '', $raw);
    return ltrim($digits, '0'); // simple normalize; keep country code if present
}
function _make_lead_id(array $row, string $tabName, int $rowNo): string {
    $email = strtolower(_first_nonempty($row, ['email','e_mail','mail']));
    $phone = _normalize_phone(_first_nonempty($row, ['phone','phone_number','contact','mobile']));
    $ctime = _first_nonempty($row, ['created_time','created at','timestamp','createdtime']);
    $parts = [$email, $phone, $ctime, $tabName, (string)$rowNo];
    $identity = implode('|', array_map(fn($v) => $v ?? '', $parts));
    $hash = sha1($identity);                  // 40 chars
    return 'H:' . $hash;                      // prefix to spot hashed IDs
}

/* -----------------------------------------------------------------------------
   2) Fetch & normalize Google Sheet rows
----------------------------------------------------------------------------- */
function getNewRows(string $sheetId, string $tabName, int $lastRow): array {
    $lastRow = max(1, $lastRow); // safety: never import row 1 (header)
    $allData = getSheetData($sheetId, "{$tabName}!1:1000"); // Fetch full tab
    if (count($allData) < 2) return []; // must have header + at least 1 row


    $headerMap = _build_header_map($allData[0]); // index -> canonical key
    $new = [];

    foreach ($allData as $i => $row) {
        if ($i + 1 <= $lastRow) continue; // already imported
        $assoc = [];
        foreach ($headerMap as $ci => $key) {
            $assoc[$key] = $row[$ci] ?? null;
        }
        $assoc['_rowNumber'] = $i + 1;
        error_log('IMPORT: row ' . $assoc['_rowNumber'] . ' sample=' . json_encode(array_slice($assoc, 0, 10)));
        $new[] = $assoc;
    }
    return $new;
}

/* -----------------------------------------------------------------------------
   3) Optional: notify admin
----------------------------------------------------------------------------- */
function notifyAdminNewLead(string $leadID, array $leadData): void {
    @mail(
        SMTP_USER,
        "New Lead: $leadID",
        print_r($leadData, true),
        "From: " . SMTP_USER . "\r\nContent-Type: text/plain"
    );
}

/* -----------------------------------------------------------------------------
   4) Get list of monitored sheets
----------------------------------------------------------------------------- */
$sources = $db->query("SELECT * FROM monitored_sheets")->fetch_all(MYSQLI_ASSOC);

/* -----------------------------------------------------------------------------
   5) Process each source
----------------------------------------------------------------------------- */
foreach ($sources as $src) {
    $sheetId = $src['sheet_id'];
    $tabName = $src['tab_name'];
    $lastRowPointer = max(1, (int)$src['last_synced_row']); // never import header
error_log("ЁЯФД Importing tab: $tabName");

$rows = getNewRows($sheetId, $tabName, $lastRowPointer);


    error_log("ЁЯза Total new rows fetched for tab $tabName: " . count($rows));
    if (empty($rows)) continue;

    // track only successfully imported rows
    $maxRowImported = (int)$lastRowPointer;

    foreach ($rows as $r) {
    // 1) lead_id from sheet or deterministic fallback
    $leadID = isset($r['id']) ? trim((string)$r['id']) : '';
    if ($leadID === '') {
        $auto = _auto_lead_id($r, $tabName);
        if ($auto !== null) $leadID = $auto;
    }
    $leadID = $db->real_escape_string($leadID);

    // 2) No ID even after fallback? skip row safely
    if ($leadID === '') {
        error_log("тЪая╕П Skipping row without ID at row {$r['_rowNumber']} keys=" . implode(',', array_keys($r)));
        continue;
    }

    // 3) Apply defaults (platform/country/name/phone mirroring, etc.)
    $r = _apply_defaults($r, $tabName);

    // 4) Duplicate check
    $existsStmt = $db->prepare("SELECT 1 FROM leads WHERE lead_id = ? LIMIT 1");
    $existsStmt->bind_param('s', $leadID);
    $existsStmt->execute();
    $existsStmt->store_result();
    if ($existsStmt->num_rows > 0) { $existsStmt->close(); continue; }
    $existsStmt->close();

    // 5) imported_at = now (we also use this as created_at fallback)
    $now = new DateTime('now', new DateTimeZone('Asia/Dubai'));
    $importedAt = $now->format('Y-m-d H:i:s');

    // 6) created_at from sheet; if missing/invalid => use importedAt
    $createdAt = null;
    $createdAtRaw = $r['created_time'] ?? null;
    if ($createdAtRaw) {
        try {
            $dt = new DateTime($createdAtRaw);
            $dt->setTimezone(new DateTimeZone('Asia/Dubai'));
            $createdAt = $dt->format('Y-m-d H:i:s');
        } catch (Exception $e) {
            error_log("тЪая╕П Invalid created_time: " . $createdAtRaw . " (using import time)");
        }
    }
    if (!$createdAt) $createdAt = $importedAt;

    // 7) Encode JSON after defaults
    $json = json_encode($r, JSON_UNESCAPED_UNICODE);

    // 8) Insert
    $stmt = $db->prepare("INSERT INTO leads (lead_id, data, tab_name, created_at, imported_at) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param('sssss', $leadID, $json, $tabName, $createdAt, $importedAt);
    $stmt->execute();


        $newLeadPK = (int)$stmt->insert_id;
        $newLeadPK = (int)$stmt->insert_id;
if ($newLeadPK > 0) {
    notifyAdminNewLead($leadID, $r);

    // 1) Log a dedicated "new_lead" notification for admins
    $changedBy = 1; // system/admin
    $role      = 'admin';
    $logStmt = $db->prepare("
        INSERT INTO lead_logs (lead_id, action, comment, changed_by, role, created_at)
        VALUES (?, 'new_lead', 'New lead synced and unassigned', ?, ?, NOW())
    ");
    $logStmt->bind_param("iis", $newLeadPK, $changedBy, $role);
    $logStmt->execute();
    $logStmt->close();

    // 2) Also log default status "New Lead"
    $defaultStatusId = 1;
    $defaultComment  = 'Default status: New Lead';
    $logStmt = $db->prepare("
        INSERT INTO lead_logs (lead_id, action, new_status_id, comment, changed_by, role, created_at)
        VALUES (?, 'status_update', ?, ?, ?, ?, NOW())
    ");
    $logStmt->bind_param("iisis", $newLeadPK, $defaultStatusId, $defaultComment, $changedBy, $role);
    $logStmt->execute();
    $logStmt->close();


            // track last successfully imported row for pointer update
            $maxRowImported = max($maxRowImported, (int)$r['_rowNumber']);
        } else {
            error_log("тЪая╕П Insert failed for lead_id={$leadID} row={$r['_rowNumber']}");
        }
        $stmt->close();
    }

    // Update last_synced_row ONLY to last successfully inserted row
    if ($maxRowImported > (int)$lastRowPointer) {
        $upd = $db->prepare("UPDATE monitored_sheets SET last_synced_row = ? WHERE id = ?");
        $srcId = (int)$src['id'];
        $upd->bind_param('ii', $maxRowImported, $srcId);
        $upd->execute();
$upd->close();
error_log("… Pointer advanced for {$tabName} to row {$maxRowImported}");

// ADD: write a single row_import log with count + new pointer
try {
    $importedCount = $maxRowImported - (int)$lastRowPointer;
    if ($importedCount > 0) {
        $sheetDbId = (int)$src['id'];
        $action = 'row_import';
        $oldV = json_encode(['last_synced_row' => (int)$lastRowPointer], JSON_UNESCAPED_UNICODE);
        $newV = json_encode(['last_synced_row' => $maxRowImported, 'imported_rows' => $importedCount], JSON_UNESCAPED_UNICODE);
        $performedBy = 0; // system
        $stmtLog = $db->prepare("
          INSERT INTO monitored_sheet_logs (sheet_id, action, old_value, new_value, performed_by, created_at)
          VALUES (?, ?, ?, ?, ?, NOW())
        ");
        $stmtLog->bind_param('isssi', $sheetDbId, $action, $oldV, $newV, $performedBy);
        $stmtLog->execute();
        $stmtLog->close();
    }
} catch (Throwable $e) {
    error_log('row_import log failed: '.$e->getMessage());
}

    } else {
        error_log("тД╣я╕П Pointer unchanged for {$tabName}; no successful inserts");
    }
}

$db->close();
