// assets/js/update_lead.js
// JSON-only. No FormData. No page reloads.

// 🍞 Toast
function showToast(message, type = 'success') {
  const toast = document.createElement('div');
  toast.className =
    `fixed top-4 right-4 px-3 py-1.5 rounded shadow text-white text-sm z-50 ${
      type === 'error' ? 'bg-red-600' : type === 'warn' ? 'bg-yellow-600' : 'bg-green-600'
    }`;
  toast.textContent = message;
  document.body.appendChild(toast);
  setTimeout(() => toast.remove(), 2400);
}

// Follow-up rules (keep in sync with backend)
const followupRequiredStatusIds = [2, 3, 4, 5, 6, 7, 8];

/* ---------------------------------------------
   TIMELINE: compact, badge-styled, utility-only
   --------------------------------------------- */
function fetchLogs(brokerId = '') {
  const leadId = parseInt(new URLSearchParams(window.location.search).get('id'));
  const url = `/api/get_lead_logs.php?lead_id=${leadId}${brokerId ? `&broker_id=${brokerId}` : ''}`;

  fetch(url)
    .then(async res => {
      let data;
      try {
        // Read once as text; parse defensively
        const text = await res.text();
        data = JSON.parse(text);
      } catch {
        const container = document.getElementById('leadLogsContainer');
        if (container) container.innerHTML = '<p class="text-red-400 text-xs">⚠️ Invalid or empty log data returned.</p>';
        return;
      }

      const container = document.getElementById('leadLogsContainer');
      if (!container) return;
      container.innerHTML = '';

      if (!data || !Array.isArray(data.logs) || !data.logs.length) {
        container.innerHTML = '<p class="text-gray-400 text-xs italic">No activity found.</p>';
        return;
      }

      // Utility class maps for compact colored badges (no @apply)
      const badgeClassMap = {
        status_update : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-blue-600/90 text-white',
        followup_update: 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-teal-600/90 text-white',
        followup      : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-teal-600/90 text-white',
        comment_only  : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-yellow-600/90 text-white',
        reassignment  : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-purple-600/90 text-white',
        unassignment  : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-red-600/90 text-white',
        assignment    : 'inline-flex items-center text-[10px] font-semibold uppercase tracking-wide rounded-full px-2 py-0.5 bg-green-600/90 text-white',
      };

      const badgeLabelMap = {
        status_update  : 'STATUS UPDATE',
        followup_update: 'FOLLOW-UP',
        followup       : 'FOLLOW-UP',
        comment_only   : 'COMMENT',
        reassignment   : 'REASSIGNED',
        unassignment   : 'UNASSIGNED',
        assignment     : 'ASSIGNED',
      };

      data.logs.forEach(log => {
        const card = document.createElement('div');
        // compact card container
        card.className = 'bg-gray-800/60 border border-gray-700 rounded-md p-3 space-y-2';

        // action → classes/label
        const actionKey   = (log.action || '').trim();
        const badgeClass  = badgeClassMap[actionKey] || 'inline-flex items-center text-[10px] font-semibold uppercase rounded-full px-2 py-0.5 bg-gray-600/90 text-white';
        const badgeLabel  = badgeLabelMap[actionKey] || 'NOTE';

        // timestamp (compact)
        const ts       = new Date(log.created_at);
        const now      = new Date();
        const isToday  = ts.toDateString() === now.toDateString();
        const timestamp = isToday
          ? `Today ${ts.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}`
          : ts.toLocaleString([], { dateStyle: 'short', timeStyle: 'short' });

        const by    = log.changed_by_name || 'Unknown';
        const role  = log.changed_by_role || 'unknown';
        let brokerName = log.broker_name || log.new_broker_name || log.old_broker_name || '—';
        if (actionKey === 'unassignment' && log.old_broker_name) brokerName = `Unassigned Broker: ${log.old_broker_name}`;
        if (actionKey === 'reassignment' && log.new_broker_name) brokerName = `Reassigned to Broker: ${log.new_broker_name}`;

        // status change line
        let statusChange = '';
        if (actionKey === 'status_update' && log.old_status_name && log.new_status_name) {
          statusChange = `
            <div class="text-[12px] text-gray-300">
              <span class="mr-1">🔄</span>${log.old_status_name} &rarr; ${log.new_status_name}
            </div>`;
        }

        // follow-up info
        let followup = '';
        if (log.followup_datetime && log.followup_datetime !== '0000-00-00 00:00:00') {
          const due = new Date(log.followup_datetime);
          const overdue = due.getTime() < Date.now();
          followup = `
            <div class="flex items-center gap-2">
              <span class="text-[12px] text-gray-300">📅 ${due.toLocaleString([], { dateStyle: 'short', timeStyle: 'short' })}</span>
              ${
                overdue
                  ? '<span class="text-[10px] font-semibold text-white bg-red-600/90 px-2 py-0.5 rounded">OVERDUE</span>'
                  : '<span class="text-[10px] font-semibold text-green-100 bg-green-700/90 px-2 py-0.5 rounded">UPCOMING</span>'
              }
            </div>`;
        }

        // comment
        let comment = (log.comment || '').trim();
        if (!comment || comment === '0') {
          comment = '<span class="italic text-gray-500 text-[13px]">No comment added</span>';
        } else {
          comment = comment.replace(/\n/g, '<br>');
        }

        // entry layout: header row (badge + time), body, meta
        card.innerHTML = `
          <div class="flex items-center">
            <span class="${badgeClass}">${badgeLabel}</span>
            <span class="ml-auto text-[11px] text-gray-400">${timestamp}</span>
          </div>

          <div class="text-[13px] text-gray-100">${comment}</div>
          ${statusChange}
          ${followup}

          <div class="flex items-center gap-2 text-[11px] text-gray-400">
            <span>Affected Broker: ${brokerName}</span>
            <span>•</span>
            <span>By ${by} (${role})</span>
          </div>
        `;

        container.appendChild(card);
      });
    })
    .catch(err => {
      console.error('Timeline error:', err);
      const container = document.getElementById('leadLogsContainer');
      if (container) container.innerHTML = '<p class="text-red-500 text-xs">Failed to load timeline.</p>';
    });
}

/* -------------------------
   FORM BINDINGS (unchanged)
   ------------------------- */
function bindLeadFormBlock(form) {
  const submitBtn    = form.querySelector('.submitBtn');
  const commentErr   = form.querySelector('.commentError');
  const followupErr  = form.querySelector('.followupError');
  const snackbar     = form.querySelector('.snackbar');

  const leadId   = +form.dataset.leadId;
  const brokerId = +form.dataset.brokerId;
  const suffix   = `${leadId}_${brokerId}`;

  const statusInput   = form.querySelector(`#status_${suffix}`);
  const followupInput = form.querySelector(`#followup_${suffix}`);
  const commentInput  = form.querySelector(`#comment_${suffix}`);

  // Quick comment → append to textarea
  const quickComment = form.querySelector(`#quickComment_${suffix}`);
  if (quickComment && commentInput) {
    quickComment.addEventListener('change', () => {
      const val = quickComment.value.trim();
      if (!val) return;
      commentInput.value = commentInput.value.trim()
        ? `${commentInput.value.trim()}\n${val}`
        : val;
      quickComment.value = '';
    });
  }

  if (!submitBtn) return;

  submitBtn.addEventListener('click', async () => {
    const oldStatusId = +form.dataset.oldStatusId || 0;
    const newStatusId = +((statusInput && statusInput.value) || 0);

    const oldFollowup = (form.dataset.oldFollowup && form.dataset.oldFollowup !== '0000-00-00 00:00:00')
      ? form.dataset.oldFollowup.trim()
      : '';
    const followup = (followupInput && followupInput.value && followupInput.value !== '0000-00-00 00:00:00')
      ? followupInput.value.trim()
      : '';

    let comment = (commentInput && commentInput.value ? commentInput.value.trim() : '');
    if (!comment) {
      if (commentErr) commentErr.classList.remove('hidden');
      return;
    } else if (commentErr) commentErr.classList.add('hidden');

    const statusChanged   = newStatusId !== oldStatusId;
    const followupChanged = followup !== oldFollowup;

    if (statusChanged && followupRequiredStatusIds.includes(newStatusId) && !followup) {
      if (followupErr) followupErr.classList.remove('hidden');
      return;
    } else if (followupErr) followupErr.classList.add('hidden');

    let action = 'comment_only';
    if (statusChanged) action = 'status_update';
    else if (followupChanged) {
      action = 'followup_update';
      if (followupRequiredStatusIds.includes(newStatusId) && !followup) {
        if (followupErr) followupErr.classList.remove('hidden');
        return;
      }
    }

    const payload = {
      lead_id: leadId,
      broker_id: brokerId,
      action,
      comment,
      old_status_id: oldStatusId,
      new_status_id: newStatusId,
      followup_datetime: followup
    };

    const originalHTML = submitBtn.innerHTML;
    submitBtn.disabled = true;
    submitBtn.innerHTML = `
      <svg class="animate-spin h-5 w-5 mx-auto text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v4l3-3-3-3v4a8 8 0 00-8 8z"></path>
      </svg>`;

    try {
      const res = await fetch('/api/log_lead_action.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });

      const contentType = res.headers.get('content-type') || '';
      const data = contentType.includes('application/json')
        ? await res.json()
        : { error: await res.text() };

      if (res.ok && data.success) {
        showToast('✅ Update saved');
        if (snackbar) {
          snackbar.textContent = 'Update saved.';
          snackbar.className = 'snackbar mt-2 text-green-400';
        }
        form.dataset.oldStatusId = String(newStatusId);
        form.dataset.oldFollowup = followup || '';
        if (commentInput) commentInput.value = '';
        fetchLogs('');
      } else {
        throw new Error(data.error || 'Unknown error');
      }
    } catch (err) {
      console.error('❌ Submission failed:', err);
      showToast('❌ Failed to save', 'error');
      if (snackbar) {
        snackbar.textContent = 'Failed to save update.';
        snackbar.className = 'snackbar mt-2 text-red-400';
      }
    } finally {
      submitBtn.disabled = false;
      submitBtn.innerHTML = originalHTML;
    }
  });
}

function initLeadFormBindings() {
  document.querySelectorAll('.leadFormBlock').forEach(bindLeadFormBlock);
}

// Unassign (admin)
function initUnassignBindings() {
  document.querySelectorAll('.unassignBtn').forEach(btn => {
    btn.addEventListener('click', async () => {
      const leadId   = +btn.getAttribute('data-lead-id');
      const brokerId = +btn.getAttribute('data-broker-id');
      if (!leadId || !brokerId) return;
      if (!confirm('Are you sure you want to unassign this broker?')) return;

      try {
        const res = await fetch('/api/log_lead_action.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            lead_id: leadId,
            broker_id: brokerId,
            action: 'unassignment',
            comment: 'Admin unassigned this broker from the lead'
          })
        });

        let data;
        const raw = await res.text();
        try { data = JSON.parse(raw); } catch (e) { data = { error: 'Invalid JSON: ' + raw }; }

        if (res.ok && data.success) {
          showToast('✅ Broker unassigned');

          const leadFormContainer = document.querySelector('#leadFormContainer');
          const assignForm = document.querySelector('#assignBrokerForm');
          const leadIdForRefresh = assignForm ? +assignForm.dataset.leadId : 0;

          if (leadFormContainer && leadIdForRefresh > 0) {
            fetch('/api/get_lead_form.php', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ lead_id: leadIdForRefresh })
            })
              .then(r => r.json())
              .then(payload => {
                if (payload && payload.success && typeof payload.html === 'string') {
                  leadFormContainer.innerHTML = payload.html;
                  initLeadFormBindings();
                  initUnassignBindings();
                  initTabBindings();
                  const firstTabBtn = document.querySelector('.tabBtn');
                  if (firstTabBtn) firstTabBtn.click();
                } else {
                  showToast('⚠️ Unassigned, but form refresh failed', 'warn');
                }
              })
              .catch(() => showToast('⚠️ Unassigned, but refresh failed', 'warn'));
          }

          fetchLogs('');
          refreshAssignContainer(leadId);
          refreshLogTabs(leadId);
        } else {
          throw new Error(data.error || 'Unknown error');
        }
      } catch (err) {
        console.error('❌ Unassign error:', err);
        showToast('❌ Failed to unassign', 'error');
      }
    });
  });
}

// Assign (admin)
function initAssignBinding() {
  const assignBtn = document.querySelector('#assignBrokerBtn');
  if (!assignBtn) return;

  assignBtn.addEventListener('click', async () => {
    const form = document.querySelector('#assignBrokerForm');
    const leadId = form ? +form.getAttribute('data-lead-id') : 0;
    const selectEl = document.querySelector('#newBrokerSelect');
    const brokerId = selectEl ? +selectEl.value : 0;

    if (!leadId || !brokerId) {
      showToast('❌ Missing lead or broker ID', 'error');
      return;
    }

    try {
      const res = await fetch('/api/log_lead_action.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          lead_id: leadId,
          new_broker_id: brokerId,
          action: 'reassignment',
          comment: 'Admin assigned broker'
        })
      });

      const contentType = res.headers.get('content-type') || '';
      const data = contentType.includes('application/json') ? await res.json() : { error: await res.text() };

      if (res.ok && data.success) {
        showToast('✅ Broker assigned');

        const assignedBrokerName =
          selectEl?.options[selectEl.selectedIndex]?.textContent?.trim() || `Broker ${brokerId}`;

        if (selectEl) {
          const optToRemove = selectEl.querySelector(`option[value="${String(brokerId)}"]`);
          if (optToRemove) optToRemove.remove();
          const hasChoices = Array.from(selectEl.options).some(o => o.value && o.value.trim() !== '');
          assignBtn.disabled = !hasChoices;
          refreshAssignContainer(leadId);
          refreshLogTabs(leadId);
        }

        const leadFormContainer = document.querySelector('#leadFormContainer');
        if (leadFormContainer) {
          fetch('/api/get_lead_form.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ lead_id: leadId })
          })
            .then(r => r.json())
            .then(payload => {
              if (payload && payload.success && typeof payload.html === 'string') {
                leadFormContainer.innerHTML = payload.html;
                initLeadFormBindings();
                initUnassignBindings();
                initTabBindings();
                initLogTabBindings();

                const tryBtns = [
                  `.tabBtn[data-broker-id="${brokerId}"]`,
                  `.tabBtn[data-broker="${brokerId}"]`,
                  `.tabBtn[data-tab*="${brokerId}"]`
                ];
                for (const sel of tryBtns) {
                  const b = document.querySelector(sel);
                  if (b) { b.click(); break; }
                }
              } else {
                showToast('⚠️ Assigned, but failed to refresh form', 'warn');
              }
            })
            .catch(() => showToast('⚠️ Assigned, but refresh failed', 'warn'));
        }

        const firstLogBtn = document.querySelector('.logTabBtn');
        const tabsWrap = firstLogBtn ? firstLogBtn.parentElement : null;
        const tabSelector = `.logTabBtn[data-broker="${brokerId}"]`;

        if (tabsWrap && !tabsWrap.querySelector(tabSelector)) {
          const newBtn = document.createElement('button');
          newBtn.className = 'logTabBtn px-2.5 py-1 rounded bg-gray-800 text-gray-300 text-xs hover:bg-gray-700';
          newBtn.setAttribute('data-broker', String(brokerId));
          newBtn.textContent = assignedBrokerName;

          newBtn.addEventListener('click', () => {
            document.querySelectorAll('.logTabBtn').forEach(b => b.classList.remove('bg-blue-700', 'text-white'));
            newBtn.classList.add('bg-blue-700', 'text-white');
            fetchLogs(String(brokerId));
          });

          tabsWrap.appendChild(newBtn);
        }

        const tryBtns = [
          `.tabBtn[data-tab*="${brokerId}"]`,
          `.logTabBtn[data-broker="${brokerId}"]`
        ];
        for (const sel of tryBtns) {
          const b = document.querySelector(sel);
          if (b) { b.click(); break; }
        }

        const allBtn = document.querySelector('.logTabBtn[data-broker=""]');
        if (allBtn) {
          document.querySelectorAll('.logTabBtn').forEach(b => b.classList.remove('bg-blue-700', 'text-white'));
          allBtn.classList.add('bg-blue-700', 'text-white');
        }
        await fetchLogs('');
        setTimeout(() => fetchLogs(''), 300);
      } else {
        throw new Error(data.error || 'Unknown error');
      }
    } catch (err) {
      console.error('❌ Assign error:', err);
      showToast('❌ Failed to assign broker', 'error');
    }
  });
}

// Broker form tabs (left block)
function initTabBindings() {
  document.querySelectorAll('.tabBtn').forEach(btn => {
    btn.addEventListener('click', () => {
      const targetTabId = btn.dataset.tab;
      document.querySelectorAll('.brokerTab').forEach(tab => tab.classList.add('hidden'));
      const activeTab = document.getElementById(targetTabId);
      if (activeTab) activeTab.classList.remove('hidden');

      document.querySelectorAll('.tabBtn').forEach(b => {
        b.classList.remove('bg-blue-600', 'text-white', 'shadow', 'shadow-blue-600/30');
        b.classList.add('bg-gray-700', 'text-gray-300');
      });
      btn.classList.add('bg-blue-600', 'text-white', 'shadow', 'shadow-blue-600/30');
      btn.classList.remove('bg-gray-700', 'text-gray-300');
    });
  });
}

// Timeline filter tabs (right block)
function initLogTabBindings() {
  document.querySelectorAll('.logTabBtn').forEach(btn => {
    btn.addEventListener('click', () => {
      const brokerId = btn.getAttribute('data-broker') || '';
      document.querySelectorAll('.logTabBtn').forEach(b => b.classList.remove('bg-blue-700', 'text-white'));
      btn.classList.add('bg-blue-700', 'text-white');
      fetchLogs(brokerId);
    });
  });
}

// Refresh assign UI HTML and rebind
async function refreshAssignContainer(leadId) {
  try {
    const res = await fetch('/api/get_assign_ui.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ lead_id: leadId })
    });
    const data = await res.json();
    if (data.success && data.html !== undefined) {
      const container = document.querySelector('#assignContainer');
      if (container) {
        container.innerHTML = data.html;
        initAssignBinding();
      }
    }
  } catch (err) {
    console.error('⚠️ Assign refresh failed:', err);
  }
}

// Refresh timeline tabs HTML and rebind
async function refreshLogTabs(leadId) {
  try {
    const res = await fetch('/api/get_lead_log_tabs.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ lead_id: leadId })
    });
    const data = await res.json();
    if (data.success && data.html) {
      const tabsContainer = document.querySelector('#logTabsContainer');
      if (tabsContainer) {
        tabsContainer.innerHTML = data.html;
        initLogTabBindings();
      }
    }
  } catch (err) {
    console.error('⚠️ Log tabs refresh failed:', err);
  }
}

// Init
initLeadFormBindings();
initUnassignBindings();
initAssignBinding();
initTabBindings();
initLogTabBindings();
fetchLogs('');
