// âœ… assets/js/sheets.js
// âœ… Run when DOM is fully loaded
document.addEventListener('DOMContentLoaded', () => {
  const sheetForm = document.getElementById('sheetForm');
  const saveBtn = document.getElementById('saveSheetBtn');
  const resetBtn = document.getElementById('resetSheetBtn');
  const cardGrid = document.getElementById('sheetsCardGrid');
    const syncBtn = document.getElementById('syncSheetsBtn'); // wire “Sync to JSON” button by id from admin/manage_sheets.php
      // Modal DOM refs for safe deletion
  const deleteModal = document.getElementById('deleteModal');
  const modalSheetInfo = document.getElementById('modalSheetInfo');
  const modalConfirmInput = document.getElementById('modalConfirmInput');
  const modalChoiceKeep = document.getElementById('modalChoiceKeep');
  const modalChoiceDelete = document.getElementById('modalChoiceDelete');
  const modalCancelBtn = document.getElementById('modalCancelBtn');
  const modalConfirmBtn = document.getElementById('modalConfirmBtn');
  let currentDelete = { id: null, tab: '' }; // holds the sheet being deleted



  // âœ… Load all existing monitored sheets and render them
  async function loadSheets() {
    const res = await fetch('/api/sheets_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ action: 'list' }) // ðŸ”‘ Backend expects 'list'
    });

    const data = await res.json();
    

    cardGrid.innerHTML = ''; // Clear existing cards

    if (Array.isArray(data?.sheets)) {
      data.sheets.forEach(sheet => renderCard(sheet));
    } else {
      showToast('No sheets found or invalid response', 'error');
    }
  }

  // âœ… Render a single card in the UI
function renderCard(sheet) {
  const card = document.createElement('div');
  card.className = 'bg-gray-800 rounded-xl p-5 text-sm text-gray-300 shadow space-y-2';

  const sheetIdDb = sheet.id; // DB row id (used by API/logs)
  const timelineId = `sheetTimeline-${sheetIdDb}`;

  card.innerHTML = `
    <div class="space-y-1 break-words">
      <p><span class="text-gray-400 font-medium">Sheet ID:</span>
         <span class="text-white break-all line-clamp-2">${sheet.sheet_id}</span></p>
      <p><span class="text-gray-400 font-medium">Tab Name:</span>
         <span class="text-white">${sheet.tab_name}</span></p>
      <p><span class="text-gray-400 font-medium">Notes:</span>
         <span class="text-white">${sheet.notes || '-'}</span></p>
      <p><span class="text-gray-400 font-medium">Last Synced Row:</span>
         <span class="text-white">${sheet.last_synced_row || 0}</span></p>
    </div>

    <div class="flex flex-wrap gap-3 pt-3">
      <button onclick="editSheet('${sheetIdDb}')"
              class="bg-yellow-600 hover:bg-yellow-700 text-white text-sm px-4 py-1.5 rounded">
        Edit
      </button>

      <button onclick="openDeleteModal(${sheetIdDb})"
              class="bg-red-600 hover:bg-red-700 text-white text-sm px-4 py-1.5 rounded">
        Delete
      </button>
  `;

  cardGrid.appendChild(card);
}


  // âœ… Reset all input fields
  function resetForm() {
    ['sheet_id', 'tab_name', 'notes', 'last_synced_row', 'sheet_id_field'].forEach(id => {
      document.getElementById(id).value = '';
    });
  }

async function validateSheet(sheet_id, tab_name) {
  const res = await fetch('/api/sheets_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      action: 'validate_sheet',
      sheet_id,
      tab_name
    })
  });
  return await res.json();
}

  // ---------- Safe delete flow (modal) ----------
  async function openDeleteModal(sheetId) {
    try {
      // Fetch latest sheet info to display exact tab name for confirmation
      const res = await fetch('/api/sheets_api.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'get', id: sheetId })
      });
      const data = await res.json();
      if (!data.success || !data.sheet) {
        showToast('Failed to load sheet', 'error');
        return;
      }

      currentDelete.id = data.sheet.id;
      currentDelete.tab = data.sheet.tab_name || '';
      modalSheetInfo.textContent = `Tab: "${currentDelete.tab}" (ID ${currentDelete.id})`;

      // reset modal controls
      modalConfirmInput.value = '';
      modalChoiceKeep.checked = true;
      modalChoiceDelete.checked = false;
      modalConfirmBtn.disabled = true;

      // show modal
      deleteModal.classList.remove('hidden');
    } catch (e) {
      console.error(e);
      showToast('Unable to open delete dialog', 'error');
    }
  }

  function closeDeleteModal() {
    deleteModal.classList.add('hidden');
    currentDelete = { id: null, tab: '' };
  }

  // enable/disable delete button based on typed confirmation
  modalConfirmInput?.addEventListener('input', () => {
    modalConfirmBtn.disabled = modalConfirmInput.value.trim() !== currentDelete.tab;
  });

  modalCancelBtn?.addEventListener('click', () => closeDeleteModal());

  modalConfirmBtn?.addEventListener('click', async () => {
    // guard: still disabled or no target
    if (modalConfirmBtn.disabled || !currentDelete.id) return;

    const cascade = modalChoiceDelete.checked === true; // true => also delete leads
    try {
      const res = await fetch('/api/sheets_api.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          action: 'delete',
          id: currentDelete.id,
          cascade_delete_leads: cascade
        })
      });
      const data = await res.json();
      if (data.success) {
        const msg = (data.deleted_leads && data.deleted_leads > 0)
          ? `Sheet deleted. ${data.deleted_leads} lead(s) removed.`
          : 'Sheet deleted. Leads kept.';
        showToast(msg, 'success');
        closeDeleteModal();
        loadSheets(); // refresh grid
        // ✅ Load sheets on page load
loadSheets().then(loadSheetsActivity);

      } else {
        showToast(data.message || 'Failed to delete', 'error');
      }
    } catch (e) {
      console.error(e);
      showToast('Network error. Try again.', 'error');
    }
  });

  // expose to inline onclick in cards
  window.openDeleteModal = openDeleteModal;


  // âœ… Handle form submission for create/update
  saveBtn.addEventListener('click', async () => {
    const payload = {
      id: document.getElementById('sheet_id_field').value.trim(),
      sheet_id: document.getElementById('sheet_id').value.trim(),
      tab_name: document.getElementById('tab_name').value.trim(),
      notes: document.getElementById('notes').value.trim(),
      last_synced_row: parseInt(document.getElementById('last_synced_row').value || '0'),
      action: 'create_or_update'
    };

    // ðŸš« Prevent empty submissions
    if (!payload.sheet_id || !payload.tab_name) {
      showToast('Sheet ID and Tab Name are required', 'error');
      return;
    }
    
    // 🛡️ Validate sheet ID and tab name before saving
const validation = await validateSheet(payload.sheet_id, payload.tab_name);
if (!validation.success) {
  showToast(validation.message || 'Invalid Sheet ID or Tab Name', 'error');
  return;
}

    
    const res = await fetch('/api/sheets_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload)
    });

    const result = await res.json();
    if (result.success) {
  showToast('Saved successfully', 'success');
  resetForm();
  await loadSheets();
  await loadSheetsActivity();
} else {

      showToast(result.message || 'Failed to save', 'error');
    }
  });

  // âœ… Reset form on button click
  resetBtn.addEventListener('click', resetForm);
  
// 🔁 Run immediate import (executes cron/import_leads.php)
if (syncBtn) {
  syncBtn.addEventListener('click', async () => {
    try {
      syncBtn.disabled = true;
      const res = await fetch('/api/sheets_api.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'import_now' })
      });
      const data = await res.json();
      if (data.success) {
        showToast('Import completed', 'success');
        await loadSheets();
        await loadSheetsActivity();
      } else {
        showToast(data.message || 'Import failed', 'error');
      }
    } catch (e) {
      console.error(e);
      showToast('Network error. Try again.', 'error');
    } finally {
      syncBtn.disabled = false;
    }
  });
}



  // âœ… Populate form for editing an existing sheet
  window.editSheet = async (sheetId) => {
  const res = await fetch('/api/sheets_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ action: 'get', id: sheetId })
  });

  const result = await res.json();
  if (!result.success || !result.sheet) {
    showToast('Failed to load sheet', 'error');
    return;
  }

  const sheet = result.sheet;

  document.getElementById('sheet_id_field').value = sheet.id; // Use internal numeric ID
  document.getElementById('sheet_id').value = sheet.sheet_id;
  document.getElementById('tab_name').value = sheet.tab_name;
  document.getElementById('notes').value = sheet.notes || '';
  document.getElementById('last_synced_row').value = sheet.last_synced_row || 0;
};




  
  // ✅ Load sheets + activity on page load
loadSheets().then(loadSheetsActivity);

});

// âœ… Reusable toast function
function showToast(message, type = 'success') {
  const toast = document.createElement('div');
  toast.className = `fixed top-5 right-5 z-50 px-4 py-2 rounded shadow text-white
                     ${type === 'success' ? 'bg-emerald-600' : 'bg-red-600'}`;
  toast.textContent = message;
  document.body.appendChild(toast);
  setTimeout(() => toast.remove(), 3000);
}

async function fetchSheetLogs(sheetDbId, limit = 50) {
  const res = await fetch('/api/sheets_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ action: 'logs', id: sheetDbId, limit })
  });
  return await res.json();
}

function renderLogRow(row) {
  // badge color by action
  const color =
    row.action === 'create' ? 'bg-green-700' :
    row.action === 'update' ? 'bg-yellow-700' :
    row.action === 'delete' ? 'bg-red-700' :
    row.action === 'sync_json' ? 'bg-blue-700' : 'bg-gray-700';

  // keep old/new short for readability
  const trim = (s) => {
    if (s == null) return '-';
    const str = String(s);
    return str.length > 160 ? str.slice(0, 160) + '…' : str;
  };

  return `
    <div class="p-2 rounded bg-gray-900 border border-gray-700">
      <div class="flex items-center gap-2">
        <span class="text-xs px-2 py-0.5 rounded text-white ${color} uppercase">${row.action}</span>
        <span class="text-xs text-gray-400">${row.at}</span>
        <span class="text-xs text-gray-300 ml-auto">${row.by}</span>
      </div>
      <div class="mt-2 grid grid-cols-2 gap-3 text-xs">
        <div><span class="text-gray-400">Old:</span> <span class="text-gray-200 break-words">${trim(row.old)}</span></div>
        <div><span class="text-gray-400">New:</span> <span class="text-gray-200 break-words">${trim(row.new)}</span></div>
      </div>
    </div>
  `;
}

async function fetchAllSheetLogs(limit = 100) {
  const res = await fetch('/api/sheets_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ action: 'logs_all', limit })
  });
  return await res.json();
}

function renderSheetLogRow(row) {
  const color =
    row.action === 'create'     ? 'bg-green-700'  :
    row.action === 'update'     ? 'bg-yellow-700' :
    row.action === 'delete'     ? 'bg-red-700'    :
    row.action === 'row_import' ? 'bg-indigo-700' :
    row.action === 'sync'       ? 'bg-blue-700'   : 'bg-gray-700';

  // Parse JSON safely
  const safeParse = (s) => {
    try { return s ? JSON.parse(s) : null; } catch { return null; }
  };
  const oldV = safeParse(row.old);
  const newV = safeParse(row.new);

  // Build a concise line
  let summary = '';
  if (row.action === 'row_import') {
    const from = oldV?.last_synced_row ?? '?';
    const to   = newV?.last_synced_row ?? '?';
    const cnt  = newV?.imported_rows ?? (to !== '?' && from !== '?' ? (to - from) : '?');
    summary = `Imported ${cnt} row(s) • Pointer ${from} → ${to}`;
  } else if (row.action === 'create') {
    summary = `Added sheet “${row.tab_name}”`;
  } else if (row.action === 'delete') {
    summary = `Deleted sheet “${row.tab_name}”`;
  } else if (row.action === 'update') {
    const oldK = oldV ? Object.keys(oldV) : [];
    const newK = newV ? Object.keys(newV) : [];
    const keys = Array.from(new Set([...oldK, ...newK])).filter(k => (oldV?.[k] ?? '') !== (newV?.[k] ?? ''));
    summary = keys.length ? `Updated: ${keys.join(', ')}` : 'Updated';
  } else if (row.action === 'sync') {
    summary = 'Synced configuration JSON';
  } else {
    summary = row.action;
  }

  // Optional raw JSON toggle
  const detailsId = `logd-${row.id}`;

  return `
    <div class="p-3 rounded bg-gray-900 border border-gray-700">
      <div class="flex items-center gap-2">
        <span class="text-xs px-2 py-0.5 rounded text-white ${color} uppercase">${row.action}</span>
        <span class="text-xs text-gray-300">Tab: <span class="font-semibold">${row.tab_name}</span></span>
        <span class="text-xs text-gray-400 ml-2">${row.at}</span>
        <span class="text-xs text-gray-300 ml-auto">${row.by}</span>
      </div>

      <div class="mt-2 text-sm text-gray-200">${summary}</div>

      <button class="mt-2 text-xs underline text-gray-400 hover:text-gray-300"
              onclick="document.getElementById('${detailsId}').classList.toggle('hidden')">
        View JSON
      </button>
      <div id="${detailsId}" class="hidden mt-2 grid grid-cols-2 gap-3 text-xs">
        <div><span class="text-gray-400">Old:</span>
          <pre class="whitespace-pre-wrap break-words text-gray-300">${row.old ? row.old : '-'}</pre>
        </div>
        <div><span class="text-gray-400">New:</span>
          <pre class="whitespace-pre-wrap break-words text-gray-300">${row.new ? row.new : '-'}</pre>
        </div>
      </div>
    </div>
  `;
}


async function loadSheetsActivity() {
  const box = document.getElementById('sheetsLogsContainer');
  if (!box) return;
  box.innerHTML = `<div class="text-gray-400 text-sm">Loading activity…</div>`;
  try {
    const data = await fetchAllSheetLogs(100);
    if (!data.success || !Array.isArray(data.logs) || data.logs.length === 0) {
      box.innerHTML = `<div class="text-gray-400 text-sm">No activity yet.</div>`;
      return;
    }
    box.innerHTML = data.logs.map(renderSheetLogRow).join('');
  } catch (e) {
    console.error(e);
    box.innerHTML = `<div class="text-red-400 text-sm">Failed to load activity.</div>`;
  }
}


