// assets/js/navbar.js
// Notifications dropdown: compact, smooth, inline-tailwind only.

let notifOffset = 0;
let notifLimit  = 10;
let notifHasMore = true;
let notifAutoTimer = null;

const notifBtn      = document.getElementById("notifBtn");
const notifBadge    = document.getElementById("notifBadge");
const notifDropdown = document.getElementById("notifDropdown");

// ---------- Inline Styles (scrollbar + dot) ----------
(function injectNotifStyles() {
  const css = `
    #notifDropdown::-webkit-scrollbar { width:0; height:0; }
    #notifDropdown:hover::-webkit-scrollbar { width:4px; height:4px; }
    #notifDropdown::-webkit-scrollbar-thumb { background:#4B5563; border-radius:9999px; }
  `;
  const style = document.createElement("style");
  style.textContent = css;
  document.head.appendChild(style);
})();

// ---------- API Helper ----------
async function api(body) {
  const res = await fetch("/api/notifications.php", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(body || {})
  });
  return res.json();
}

// ---------- Time ago ----------
function timeAgo(ts) {
  const now = new Date();
  const then = new Date(ts);
  const diff = Math.floor((now - then) / 1000);
  if (diff < 60)   return `${diff}s ago`;
  if (diff < 3600) return `${Math.floor(diff / 60)}m ago`;
  if (diff < 86400)return `${Math.floor(diff / 3600)}h ago`;
  return `${Math.floor(diff / 86400)}d ago`;
}

// ---------- Badge Poll ----------
async function pollUnread() {
  try {
    const data = await api({ mode: "count" });
    if (data.success) {
      if (data.unread_count > 0) {
        notifBadge.textContent = String(data.unread_count);
        notifBadge.classList.remove("hidden");
      } else {
        notifBadge.textContent = "";
        notifBadge.classList.add("hidden");
      }
    }
  } catch (e) { console.error("notif count", e); }
}
setInterval(pollUnread, 60000);
pollUnread();

// ---------- Header ----------
function renderHeader() {
  const header = document.createElement("div");
  header.className =
    "flex justify-between items-center px-2 py-1 border-b border-gray-700 text-gray-200 sticky top-0 bg-gray-800 text-sm";
  header.innerHTML = `
    <span class="font-semibold">Notifications</span>
    <div class="space-x-2">
      <button id="notifRefresh" class="text-xs text-blue-400 hover:underline">Refresh</button>
      <button id="notifMarkAll" class="text-xs text-blue-400 hover:underline">Mark all as read</button>
    </div>
  `;
  notifDropdown.appendChild(header);
document.getElementById("notifRefresh").onclick = (e) => {
  e.preventDefault();  // prevent default click from blurring/closing
  e.stopPropagation(); // keep dropdown open
  loadList(true);
};
document.getElementById("notifMarkAll").onclick = (e) => {
  e.preventDefault();  // prevent default closing behavior
  e.stopPropagation(); // keep dropdown open
  markAllRead();
};

}

// ---------- Load More ----------
function renderLoadMore() {
  const btn = document.createElement("button");
  btn.id = "notifLoadMore";
  btn.className = "w-full text-center py-1 text-xs text-blue-400 hover:underline bg-gray-900";
  btn.textContent = "Load more";
  btn.onclick = () => loadList(false);
  notifDropdown.appendChild(btn);
}

// ---------- List ----------
async function loadList(reset) {
  if (reset) {
    notifOffset = 0;
    notifHasMore = true;
    notifDropdown.innerHTML = "";
    renderHeader();
  }
  if (!notifHasMore) return;

  try {
    const data = await api({ mode: "list", limit: notifLimit, offset: notifOffset });
    if (!data.success) return;

    // Empty state
    if (reset && data.notifications.length === 0) {
      const empty = document.createElement("div");
      empty.className = "px-2 py-3 text-center text-xs text-gray-400";
      empty.textContent = "No notifications in the last 10 days";
      notifDropdown.appendChild(empty);
    }

    for (const n of data.notifications) {
      const item = document.createElement("div");
      item.className =
        "flex items-start px-2 py-1.5 border-b border-gray-700 cursor-pointer transition " +
        (n.is_read ? "" : "bg-gray-800/70 hover:bg-gray-700");

      // Dot
      const dot = document.createElement("div");
      dot.className = "w-2 h-2 rounded-full bg-blue-500 mt-2 mr-2 flex-shrink-0";
      if (n.is_read) dot.style.visibility = "hidden";

      const content = document.createElement("div");
      content.className = "flex-1";

      const msgLine = document.createElement("div");
      msgLine.className =
        "text-sm " + (n.is_read ? "text-gray-300 font-normal" : "text-white font-semibold");
      msgLine.textContent = n.message;

      const meta = document.createElement("div");
      meta.className = "flex justify-between text-xs text-gray-400";
      meta.innerHTML = `<span>By ${n.actor_name} (${n.actor_role})</span><span>${timeAgo(n.time)}</span>`;

      content.appendChild(msgLine);
      content.appendChild(meta);

      item.appendChild(dot);
      item.appendChild(content);

      // Optimistic mark-as-read
      item.onclick = async (e) => {
        e.stopPropagation();
        item.classList.remove("bg-gray-800/70");
        msgLine.classList.remove("text-white", "font-semibold");
        msgLine.classList.add("text-gray-300", "font-normal");
        dot.style.visibility = "hidden";

        let current = parseInt(notifBadge.textContent || "0", 10);
        if (current > 0) {
          const newVal = current - 1;
          notifBadge.textContent = newVal ? String(newVal) : "";
          if (!newVal) notifBadge.classList.add("hidden");
        }

        try {
          await api({ action: "mark_one_read", type: n.type, related_id: n.related_id });
        } catch (e) {
          console.error("mark_one_read failed", e);
        }

        setTimeout(() => {
          window.location.href = `/admin/update_lead.php?id=${n.related_id}`;
        }, 200);
      };

      notifDropdown.appendChild(item);
    }

    notifOffset += data.notifications.length;
    notifHasMore = data.has_more;

    const oldMore = document.getElementById("notifLoadMore");
    if (oldMore) oldMore.remove();
    if (notifHasMore) renderLoadMore();

    if (data.unread_count > 0) {
      notifBadge.textContent = String(data.unread_count);
      notifBadge.classList.remove("hidden");
    } else {
      notifBadge.textContent = "";
      notifBadge.classList.add("hidden");
    }
  } catch (e) {
    console.error("notif list", e);
  }
}

// ---------- Open/Close ----------
notifBtn?.addEventListener("click", async (e) => {
  e.stopPropagation();
  const isHidden = notifDropdown.classList.contains("hidden");

  if (isHidden) {
    notifDropdown.classList.remove("hidden");
    notifDropdown.classList.add("opacity-0", "translate-y-2", "scale-95");
    setTimeout(() => {
      notifDropdown.classList.remove("opacity-0", "translate-y-2", "scale-95");
      notifDropdown.classList.add("opacity-100", "translate-y-0", "scale-100");
    }, 10);

    await loadList(true);

    if (notifAutoTimer) clearInterval(notifAutoTimer);
    notifAutoTimer = setInterval(() => loadList(true), 30000);
  } else {
    closeDropdown();
  }
});

// Outside click & Esc
document.addEventListener("click", (e) => {
  if (!notifDropdown.classList.contains("hidden") && !notifDropdown.contains(e.target) && e.target !== notifBtn) {
    closeDropdown();
  }
});
document.addEventListener("keydown", (e) => {
  if (e.key === "Escape" && !notifDropdown.classList.contains("hidden")) closeDropdown();
});

function closeDropdown() {
  notifDropdown.classList.add("opacity-0", "translate-y-2", "scale-95");
  setTimeout(() => {
    notifDropdown.classList.add("hidden");
    notifDropdown.classList.remove("opacity-100", "translate-y-0", "scale-100");
  }, 150);
  if (notifAutoTimer) {
    clearInterval(notifAutoTimer);
    notifAutoTimer = null;
  }
}

// ---------- Mark all ----------
async function markAllRead() {
  try {
    const data = await api({ action: "mark_all_read" });
    if (data.success) {
      notifBadge.textContent = "";
      notifBadge.classList.add("hidden");
      await loadList(true);
    }
  } catch (e) {
    console.error("mark all", e);
  }
}
window.markAllRead = markAllRead;
