// leads.js — simple <select> scope, stable fetch, badges, filters
// -----------------------------------------------------------------------------

let currentTab = null;          // 'unassigned' | 'all' | <broker_id>
let searchMode = false;
let currentPage = 1;
let pageSize = 10;
let currentSort = 'newest';
let filters = {};

let inflight; // AbortController for fetch cancellation

// Clamp currentPage to valid range; return true if it changed
function clampPage(total) {
  const maxPage = Math.max(1, Math.ceil((total || 0) / pageSize));
  const before = currentPage;
  if (currentPage > maxPage) currentPage = maxPage;
  if (currentPage < 1) currentPage = 1;
  return before !== currentPage;
}

// DOM refs
const leadsCardsContainer = document.getElementById('leadsCardsContainer');
const leadsEmptyState     = document.getElementById('leadsEmptyState');
const leadsPagination     = document.getElementById('leadsPagination');

const searchInput     = document.getElementById('leadsSearch');
const filterFields    = document.querySelectorAll('[data-filter-type]');
const statusField     = document.getElementById('leadsStatus');
const followupField   = document.getElementById('leadsFollowups');
const roleField       = document.getElementById('leadsUserRole');

const sortField       = document.getElementById('leadsSort');
const paginationField = document.getElementById('leadsPerPage');

// NEW: plain select (no search)
const brokerSelector  = document.getElementById('leadsTabSelect');

// -----------------------------------------------------------------------------

// Read query params and apply initial tab + filters
function applyDeepLinkFromURL() {
  const q = new URLSearchParams(window.location.search);

  // tab: 'unassigned' | 'all' | <broker_id>
  const tabParam = q.get('tab');
  if (tabParam) currentTab = tabParam;

  // Supported filters from dashboard deep links
  const map = {
  range: 'range',           // today | 7d | 30d
  followup: 'followups',    // ✅ match data-filter-type="followups"
  status: 'status_id',
  platform: 'platform',
  country: 'country'
};

  Object.keys(map).forEach(k => {
    const v = q.get(k);
    if (v !== null) filters[map[k]] = v;
  });

  // Push values into filter fields if present so UI matches state
  filterFields.forEach(field => {
    const key = field.dataset.filterType;
    if (key && filters[key] !== undefined) {
      field.value = String(filters[key]);
    }
  });

  // Keep the scope <select> in sync if present
  if (brokerSelector && currentTab) {
    brokerSelector.value = currentTab;
  }
}


function updateStatusFollowupAvailability() {
  const isAdmin = roleField ? roleField.value === 'admin' : false;
  const onUnassigned = currentTab === 'unassigned';
  const shouldDisable = (isAdmin && onUnassigned) || searchMode;

  [statusField, followupField].forEach(el => {
    if (!el) return;
    el.disabled = !!shouldDisable;
    el.classList.toggle('opacity-50', !!shouldDisable);
    el.classList.toggle('cursor-not-allowed', !!shouldDisable);

    // Clear stale values so they don't leak into payload
    if (shouldDisable && el.value !== '') {
      const key = el.dataset.filterType;
      el.value = '';
      if (key) filters[key] = '';
    }
  });
}

// Switch scope and refresh
function setCurrentTab(nextTab) {
  currentTab = nextTab;     // 'unassigned' | 'all' | <broker_id>
  currentPage = 1;

  // keep select in sync if changed programmatically
  if (brokerSelector && brokerSelector.value !== currentTab) {
    brokerSelector.value = currentTab;
  }

  updateStatusFollowupAvailability();
  setResultCountLoading();
  fetchAndRenderLeads();
}

// -----------------------------------------------------------------------------

document.addEventListener('DOMContentLoaded', () => {
  // Apply deep-link params first
  applyDeepLinkFromURL();

  // Scope select
  if (brokerSelector) {
    brokerSelector.addEventListener('change', () => {
      setCurrentTab(brokerSelector.value);
    });
    // Initial scope respects deep-link tab if present
    const initial = currentTab || brokerSelector.value || 'unassigned';
    setCurrentTab(initial);
  } else {
    // Fallback when select missing (respect deep-link if present)
    setCurrentTab(currentTab || 'all');
  }


  // Search
  if (searchInput) {
    searchInput.addEventListener('input', () => {
      searchMode = searchInput.value.trim().length > 0;
      currentPage = 1;
      updateStatusFollowupAvailability();
      setResultCountLoading();
      fetchAndRenderLeads();
    });
  }

  // Filters
  filterFields.forEach(field => {
    field.addEventListener('change', () => {
      filters[field.dataset.filterType] = field.value;
      currentPage = 1;
      setResultCountLoading();
      fetchAndRenderLeads();
    });
  });

  // Sort
  if (sortField) {
    currentSort = (sortField.value || 'newest').toLowerCase();
    sortField.addEventListener('change', () => {
      currentSort = sortField.value;
      currentPage = 1;
      fetchAndRenderLeads();
    });
  }

  // Page size
  if (paginationField) {
    pageSize = parseInt(paginationField.value || '10', 10);
    paginationField.addEventListener('change', () => {
      pageSize = parseInt(paginationField.value, 10);
      currentPage = 1;
      setResultCountLoading();
      fetchAndRenderLeads();
    });
  }

  populateCountryFilter();
});

// -----------------------------------------------------------------------------

async function populateCountryFilter() {
  try {
    const res = await fetch('/api/leads_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ action: 'get_countries' })
    });
    const data = await res.json();
    if (res.ok && data.ok && Array.isArray(data.countries)) {
      const countrySelect = document.getElementById('leadsCountry');
      if (countrySelect) {
        countrySelect.innerHTML = '<option value="">All countries</option>';
        data.countries.forEach(c => {
          const o = document.createElement('option');
          o.value = c;
          o.textContent = c;
          countrySelect.appendChild(o);
        });
      }
    }
  } catch (err) {
    console.error('Failed to load countries:', err);
  }
}

// -----------------------------------------------------------------------------

async function fetchAndRenderLeads() {
  try {
    // cancel any in-flight request
    if (inflight) inflight.abort();
    inflight = new AbortController();

    const payload = {
      tab: currentTab,
      search: searchInput ? searchInput.value.trim() : '',
      search_mode: searchMode,
      filters,
      sort: currentSort,
      page: currentPage,
      page_size: pageSize
    };

    const res = await fetch('/api/leads_api.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload),
      signal: inflight.signal
    });

    const data = await res.json();
    if (!res.ok || !data.ok) throw new Error(data.error || 'Failed to load leads');

    // Admin "All Leads" — keep compatibility structures
    if (currentTab === 'all') {
      data.data.forEach(lead => {
        if (Array.isArray(lead.broker_data)) {
          lead.status_array = lead.broker_data.map(b => `${b.broker_initials} ${b.status || '-'}`);
          lead.followup_array = lead.broker_data.map(b => ({
            broker_initials: b.broker_initials, followup: b.followup
          }));
        }
      });
    }

    // Clamp page; only refetch if clamping changed the page
    if (clampPage(data.total)) {
      setResultCountLoading();
      fetchAndRenderLeads();
      return;
    }

    renderLeads(data.data);
    renderPagination(data.total);
    updateResultCount(data.total);

  } catch (err) {
    if (err.name === 'AbortError') return; // expected
    console.error('Error fetching leads:', err);
    leadsCardsContainer.innerHTML = '';
    if (leadsEmptyState) leadsEmptyState.classList.remove('hidden');
    updateResultCount(0);
    const titleElement = document.getElementById('emptyStateTitle');
    if (titleElement) {
      let titleText = 'No leads found';
      if (currentTab === 'unassigned')       titleText = 'No unassigned leads';
      else if (currentTab === 'all')         titleText = 'No leads found in All Leads';
      else if (!isNaN(parseInt(currentTab))) titleText = 'No leads found for this broker';
      else if (searchMode)                   titleText = 'No search results found';
      titleElement.textContent = titleText;
    }
  }
}

// -----------------------------------------------------------------------------

function renderBadge(label, color = 'gray', icon = '', tooltip = '') {
  const colorClass = ({
    gray: 'bg-gray-700',
    red: 'bg-red-700',
    yellow: 'bg-yellow-600',
    indigo: 'bg-indigo-700',
    blue: 'bg-blue-700',
    green: 'bg-green-700',
    purple: 'bg-purple-700'
  }[color]) || 'bg-gray-700';

  return `<span class="inline-flex items-center gap-1 text-xs sm:text-sm px-2 py-1 rounded-full ${colorClass} text-white"
    ${tooltip ? `title="${tooltip}"` : ''}>
    ${icon ? `<i class="fas fa-${icon}"></i>` : ''}${label}
  </span>`;
}

function formatFollowupBadge(dateStr) {
  if (!dateStr) return renderBadge('-', 'gray', 'clock');
  const followup = new Date(dateStr);
  const now = new Date();
  const diff = Math.floor((followup - now) / (1000 * 60 * 60 * 24));
  const tooltip = followup.toLocaleString();

  if (followup < now) return renderBadge('Overdue', 'red', 'exclamation-circle', tooltip);
  if (diff === 0)     return renderBadge('Today', 'yellow', 'calendar-day', tooltip);
  if (diff === 1)     return renderBadge('Tomorrow', 'indigo', 'calendar-plus', tooltip);
  if (diff <= 3)      return renderBadge('3d', 'blue', 'calendar-check', tooltip);
  if (diff <= 7)      return renderBadge('7d', 'green', 'calendar-week', tooltip);
  return renderBadge('+7d', 'gray', 'calendar', tooltip);
}

function formatMaskedPhone(phone) {
  return phone ? phone.replace(/\d(?=\d{2})/g, '*') : '-';
}
function formatMaskedEmail(email) {
  return email ? email.replace(/(.{2}).+(@.+)/, '$1****$2') : '-';
}
function translatePlatform(p) {
  switch ((p || '').toLowerCase()) {
    case 'fb': return 'Facebook';
    case 'ig': return 'Instagram';
    case 'gg': return 'Google';
    case 'unknown': return 'Unknown';
    default:   return p || 'Unknown';
  }
}


function renderLeadCard(lead) {
  const isUnassigned = currentTab === 'unassigned';
  const isAll        = currentTab === 'all';

  let statusHtml = '';
  let followupHtml = '';
  let extraBrokerRowHtml = '';

  if (isAll) {
    const brokerData = Array.isArray(lead.broker_data) ? lead.broker_data : [];

    // true unassigned = no broker records at all
    const isAllUnassigned = !brokerData.length;

    if (isAllUnassigned) {
      statusHtml = renderBadge('Unassigned', 'gray', 'user-slash');
      followupHtml = '';
    } else {
      const brokerRows = brokerData.map((b, i) => {
        const brokerName  = b.broker_name || 'Broker';

        // Decide badge style
        const statusBadge = b.status
          ? renderBadge(
              b.status,
              b.status === 'New Lead' ? 'blue' : 'purple',
              b.status === 'New Lead' ? 'bolt' : 'tag'
            )
          : '';

        const followupBadge = b.followup ? formatFollowupBadge(b.followup) : '';

        if (!statusBadge && !followupBadge) return '';
        return `<div class="flex flex-wrap items-center gap-2 ${i > 0 ? 'mt-2' : ''}">
          <span class="font-medium text-sm text-white whitespace-nowrap">${brokerName}:</span>
          ${statusBadge}
          ${followupBadge}
        </div>`;
      });
      statusHtml = brokerRows.join('');
      followupHtml = '';
    }
  } else if (isUnassigned) {
    // explicit unassigned tab
    statusHtml = renderBadge('Unassigned', 'gray', 'user-slash');
    followupHtml = '';
  } else {
    // broker-specific tab
    statusHtml = renderBadge(
      lead.status_name || '-',
      lead.status_name === 'New Lead' ? 'blue' : 'green',
      lead.status_name === 'New Lead' ? 'bolt' : 'check'
    );
    followupHtml = lead.followup_datetime ? formatFollowupBadge(lead.followup_datetime) : '';
  }

  const platformCode  = (lead.platform || '').toLowerCase();
  const platformLabel = translatePlatform(platformCode);
  const platformIcon  =
    platformCode === 'fb' ? '<i class="fab fa-facebook-f mr-1"></i>' :
    platformCode === 'ig' ? '<i class="fab fa-instagram mr-1"></i>' :
    platformCode === 'gg' ? '<i class="fab fa-google mr-1"></i>' :
    '<i class="fas fa-bullhorn mr-1"></i>';

  const name    = (lead.lead_name && lead.lead_name.trim()) ? lead.lead_name : 'Unnamed Lead';
  const phone   = formatMaskedPhone(lead.lead_phone);
  const email   = formatMaskedEmail(lead.lead_email);
  const country = lead.country || '-';
  const project = lead.tab_name || '-';
  const created = lead.created_at || '-';

  const card = document.createElement('div');
  card.className = 'rounded-lg bg-gray-800 border border-gray-700 p-4 shadow hover:shadow-lg transition text-xs text-gray-300 flex flex-col gap-2';

  const header = document.createElement('div');
  header.className = 'flex items-center justify-between gap-3';
  header.innerHTML = `
    <h3 class="font-semibold text-gray-100 flex items-center gap-1 truncate">
      <i class="fas fa-user-circle text-blue-400"></i>
      <span class="truncate" title="${name}">${name}</span>
    </h3>
    <span class="text-[11px] text-gray-400 whitespace-nowrap">
      <i class="far fa-clock mr-1"></i>${created}
    </span>`;

  const idRow = document.createElement('div');
  idRow.className = 'text-[11px] text-blue-300 font-mono ml-1';
  idRow.innerHTML = `Lead ID: ${lead.id}`;

  const facts = document.createElement('div');
  facts.className = 'grid grid-cols-2 gap-x-4 gap-y-1';
  facts.innerHTML = `
    <div class="truncate" title="${phone}"><i class="fas fa-phone-alt mr-1"></i>${phone}</div>
    <div class="truncate" title="${email}"><i class="fas fa-envelope mr-1"></i><span class="truncate inline-block max-w-[180px]">${email}</span></div>
    <div class="truncate" title="${country}"><i class="fas fa-flag mr-1"></i>${country}</div>
    <div class="truncate" title="${project}"><i class="fas fa-building mr-1"></i>${project}</div>
    <div class="col-span-2 truncate" title="${platformLabel}">${platformIcon}${platformLabel}</div>`;

  const badges = document.createElement('div');
  badges.className = 'flex flex-col gap-1 pt-1';

  if (!isUnassigned) {
    const statusRow = document.createElement('div');
    statusRow.className = 'flex flex-col gap-1';
    statusRow.innerHTML = `<div class="text-gray-400 font-semibold text-[13px]">Status & Follow-up:</div>${statusHtml}`;
    badges.appendChild(statusRow);

    if (extraBrokerRowHtml) {
      const brokerRow = document.createElement('div');
      brokerRow.innerHTML = extraBrokerRowHtml;
      badges.appendChild(brokerRow);
    }
  } else {
    const unassignedRow = document.createElement('div');
    unassignedRow.className = 'flex items-center gap-2';
    unassignedRow.innerHTML = `<i class="fas fa-user-slash text-gray-400"></i>${statusHtml}`;
    badges.appendChild(unassignedRow);
  }

  if (!isUnassigned && !isAll && followupHtml) {
    const fuRow = document.createElement('div');
    fuRow.className = 'flex flex-col gap-1';
    fuRow.innerHTML = `<div class="text-gray-400 font-semibold text-[13px]">Follow-up:</div>${followupHtml}`;
    badges.appendChild(fuRow);
  }

  const footer = document.createElement('div');
  footer.className = 'pt-2 text-right';
  footer.innerHTML = `<a href="/admin/update_lead.php?id=${lead.id}" class="text-blue-400 hover:underline inline-flex items-center gap-1">
    <i class="fas fa-edit"></i><span>View / Update</span>
  </a>`;

  card.appendChild(header);
  card.appendChild(idRow);
  card.appendChild(facts);
  card.appendChild(badges);
  card.appendChild(footer);

  return card;
}


function renderLeads(leads) {
  leadsCardsContainer.innerHTML = '';
  if (!leads || leads.length === 0) {
    if (leadsEmptyState) leadsEmptyState.classList.remove('hidden');
    return;
  }
  if (leadsEmptyState) leadsEmptyState.classList.add('hidden');

  leads.forEach(lead => {
    const card = renderLeadCard(lead);
    leadsCardsContainer.appendChild(card);
  });
}

function renderPagination(total) {
  leadsPagination.innerHTML = '';
  if (total <= pageSize) return;
  const totalPages = Math.ceil(total / pageSize);
  for (let p = 1; p <= totalPages; p++) {
    const btn = document.createElement('button');
    btn.textContent = p;
    btn.className = `mx-1 px-2 py-1 rounded ${p === currentPage ? 'bg-blue-600 text-white' : 'bg-gray-700 text-gray-300'}`;
    btn.addEventListener('click', () => {
      currentPage = p;
      setResultCountLoading();
      fetchAndRenderLeads();
    });
    leadsPagination.appendChild(btn);
  }
}

function setResultCountLoading() {
  const el = document.getElementById('leadsResultCount');
  if (el) el.textContent = 'Loading…';
}

function updateResultCount(total) {
  const el = document.getElementById('leadsResultCount');
  if (!el) return;
  if (!total || total <= 0) { el.textContent = '0 results'; return; }
  const start = (currentPage - 1) * pageSize + 1;
  const end   = Math.min(currentPage * pageSize, total);
  el.textContent = `Showing ${start}–${end} of ${total}`;
}
