// assets/js/dashboard.js
// Fetch role-aware counts and render tiles. JSON-only; no FormData.

async function loadCounts() {
  const res = await fetch('/api/dashboard_counts.php', {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({ filters: {} })
  });

  if (!res.ok) throw new Error(`HTTP ${res.status}`);
  const json = await res.json();
  if (!json.ok) throw new Error(json.error || 'Failed to load');

 const d = json.data || {};
// ADD: expose role for feature gating
window.APP_ROLE = json.role || 'broker';

// New leads
setText('newToday', d.new_leads?.today ?? 0);

  setText('new7d',    d.new_leads?.['7d'] ?? 0);
  setText('new30d',   d.new_leads?.['30d'] ?? 0);

  // Unassigned (admin only; API returns null for brokers)
  if (d.unassigned === null || d.unassigned === undefined) {
    const unassignedCard = document.getElementById('unassigned')?.closest('.bg-gray-800');
    if (unassignedCard) unassignedCard.classList.add('hidden'); // hide card when broker
  } else {
    setText('unassigned', d.unassigned);
  }

  // Follow-ups
  setText('fuOverdue', d.followups?.overdue ?? 0);
  setText('fuToday',   d.followups?.today ?? 0);
  setText('fu3d',      d.followups?.next_3_days ?? 0);
  setText('fu7d',      d.followups?.next_7_days ?? 0);

// Status distribution list → clickable to status filter
renderStatusList('statusList', d.status || []);

// Platform distribution list → clickable to platform filter
renderPlatformList('platformList', d.platforms || []);

  // Overdue alert banner
  const nOver = d.followups?.overdue ?? 0;
  const alertBox = document.getElementById('overdueAlert');
  const alertMsg = document.getElementById('overdueMsg');
  if (alertBox) {
    if (nOver > 0) {
      alertBox.classList.remove('hidden');
      if (alertMsg) alertMsg.textContent = `⚠️ You have ${nOver} overdue follow-ups`;
    } else {
      alertBox.classList.add('hidden');
    }
  }

  // ⬇️ NEW: expose role and hide leaderboard card for brokers
  window.IS_ADMIN = (json.role === 'admin');
  const lbCard = document.getElementById('leaderboardCard');
  if (lbCard && !window.IS_ADMIN) lbCard.classList.add('hidden');
}


// --- Timezone helpers (Asia/Dubai) for chart labels only ---
const DUBAI_TZ = 'Asia/Dubai';
function labelToDubai(dateStrYmd) {
  // Input "YYYY-MM-DD" (UTC day). Render label in Dubai's local calendar day.
  const d = new Date(dateStrYmd + 'T00:00:00Z');
  return new Intl.DateTimeFormat(undefined, { timeZone: DUBAI_TZ, month:'short', day:'2-digit' }).format(d);
}


async function loadActivity() {
  const res = await fetch('/api/dashboard_activity.php', {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({ limit: 20 })
  });
  if (!res.ok) throw new Error(`HTTP ${res.status}`);
  const json = await res.json();
  if (!json.ok) throw new Error(json.error || 'Failed to load activity');

  renderActivity('activityList', json.data || []);
}

async function loadFollowups() {
  const res = await fetch('/api/dashboard_followups.php', {
    method: 'POST',
    headers: {'Content-Type': 'application/json'},
    body: JSON.stringify({ limit: 10 })
  });
  if (!res.ok) throw new Error(`HTTP ${res.status}`);
  const json = await res.json();
  if (!json.ok) throw new Error(json.error || 'Failed to load followups');

  renderFollowups(json.data || { overdue: [], today: [] });
}

async function loadTrends() {
  if (typeof Chart === 'undefined') return;
  const res = await fetch('/api/dashboard_trends.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({}) });
 
  const j = await res.json(); if (!j.ok) throw new Error('trends failed');
  const labels = j.data.labels.map(labelToDubai);
  const series = j.data.series;

  const ctx = document.getElementById('leadsTrend');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'line',
    data: { labels, datasets: [{ label: 'Leads', data: series, tension: .25, fill: true }]},
    options: {
      responsive: true,
      plugins: { legend: { display: false } },
      scales: { y: { beginAtZero: true } }
    }
  });
}

async function loadFunnel() {
  const res = await fetch('/api/dashboard_funnel.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({}) });
  const j = await res.json(); if (!j.ok) throw new Error('funnel failed');
  const labels = ['New','Contacted','Engaged','Qualified','Booked','Closed'];
  const order  = [1,3,4,6,9,10];
  const counts = order.map(id => (j.data.find(x=>x.status_id===id)?.count || 0));

  const ctx = document.getElementById('funnelChart');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'bar',
    data: { labels, datasets: [{ label: 'Leads', data: counts }]},
    options: {
      indexAxis: 'y',
      plugins: { legend: { display:false } },
      scales: { x: { beginAtZero: true } }
    }
  });
}

async function loadLeaderboard() {
  // Do nothing if the card/table is not on this page
  const tb = document.querySelector('#leaderboardTable tbody');
  if (!tb) return;

  const res = await fetch('/api/dashboard_leaderboard.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ days:30 }) });
  const j = await res.json(); if (!j.ok) throw new Error('leaderboard failed');

    tb.innerHTML = '';

  // sort by Assigned desc, then FU Done desc
  j.data.sort(
    (a, b) =>
      (b.assigned - a.assigned) ||
      (b.followups_completed - a.followups_completed)
  );

    j.data.forEach(r => {
    const tr = document.createElement('tr');
    tr.innerHTML = `
      <td class="py-2 px-2 pr-2">${escapeHtml(r.broker_name || ('Broker #'+r.broker_id))}</td>
      <td class="text-right px-2">${r.assigned}</td>
      <td class="text-right px-2">${r.followups_completed}</td>
      <td class="text-right px-2">${r.qualified}</td>
      <td class="text-right px-2">${r.booked}</td>
      <!-- HIDE on xs to match header -->
      <td class="text-right px-2 hidden sm:table-cell">${r.closed}</td>`;
    tb.appendChild(tr);
  });

}

async function loadHeatmap() {
  const res = await fetch('/api/dashboard_heatmap.php', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ days:42 }) });
  const j = await res.json(); if (!j.ok) throw new Error('heatmap failed');
  const labels = j.data.labels.map(labelToDubai);
  const series = j.data.series;

  const ctx = document.getElementById('heatmapChart');
  if (!ctx) return;
  // Simple "heatmap" via bar chart (day bins)
  new Chart(ctx, {
    type: 'bar',
    data: { labels, datasets: [{ label: 'Follow-ups', data: series }]},
    options: {
      plugins: { legend: { display:false } },
      scales: { y: { beginAtZero: true } }
    }
  });
}


// run loaders (call leaderboard ONLY for admins, after we know the role from loadCounts)
(async function initDashboard() {
  try {
    await loadCounts(); // sets window.IS_ADMIN and hides card for brokers
    const tasks = [loadActivity(), loadFollowups(), loadTrends(), loadFunnel(), loadHeatmap()];
    if (window.IS_ADMIN) tasks.push(loadLeaderboard());
    await Promise.all(tasks);
  } catch (err) {
    console.error('Dashboard load failed:', err);
  }
})();





function setText(id, val) {
  const el = document.getElementById(id);
  if (el) el.textContent = String(val);
}

function renderKVList(containerId, pairs) {
  const ul = document.getElementById(containerId);
  if (!ul) return;
  ul.innerHTML = '';
  pairs.forEach(([label, count]) => {
    const li = document.createElement('li');
    li.className = 'flex justify-between items-center bg-gray-900 rounded-lg px-3 py-2';
    const a = document.createElement('span'); a.textContent = label ?? '-';
    const b = document.createElement('span'); b.textContent = count ?? 0; b.className = 'font-semibold';
    li.append(a, b);
    ul.appendChild(li);
  });
}

function normalizePlatform(p) {
  // Map raw codes and junk values to clean labels.
  if (!p) return 'Unknown';
  const x = String(p).trim().toLowerCase();
  if (x === 'fb') return 'Facebook';
  if (x === 'ig') return 'Instagram';
  if (x === 'gg' || x === 'google') return 'Google';
  if (x === '-' || x === 'platform' || x === 'unknown' || x === 'n/a') return 'Unknown';
  return p;
}



function renderStatusList(containerId, rows) {
  const ul = document.getElementById(containerId);
  if (!ul) return;
  ul.innerHTML = '';

  const max = Math.max(1, ...rows.map(r => Number(r.count || 0)));
  rows.forEach(s => {
    const count = Number(s.count || 0);
    const pct = Math.round((count / max) * 100);

    const li = document.createElement('li');
    li.className = 'bg-gray-900 rounded-lg px-3 py-2';

    const a = document.createElement('a');
    a.href = `/admin/leads.php?tab=all&status=${encodeURIComponent(s.id)}`;
    a.className = 'block hover:underline';

    // Top row: label + count
    const top = document.createElement('div');
    top.className = 'flex justify-between items-center';
    const left = document.createElement('span'); left.textContent = s.name ?? '-';
    const right = document.createElement('span'); right.textContent = count; right.className = 'font-semibold';
    top.append(left, right);

    // Bar
    const barWrap = document.createElement('div');
    barWrap.className = 'mt-2 h-2 bg-gray-800 rounded';
    const bar = document.createElement('div');
    bar.className = 'h-2 rounded bg-blue-600';
    bar.style.width = `${pct}%`;
    barWrap.appendChild(bar);

    a.append(top, barWrap);
    li.appendChild(a);
    ul.appendChild(li);
  });
}

function renderPlatformList(containerId, rows) {
  const ul = document.getElementById(containerId);
  if (!ul) return;
  ul.innerHTML = '';

  // Normalize labels and compute max
  const data = rows.map(p => {
    const label = normalizePlatform(p.platform);
    return { label, count: Number(p.count || 0) };
  });
  const max = Math.max(1, ...data.map(d => d.count));

  data.forEach(d => {
    const pct = Math.round((d.count / max) * 100);

    const li = document.createElement('li');
    li.className = 'bg-gray-900 rounded-lg px-3 py-2';

    const a = document.createElement('a');
    a.href = `/admin/leads.php?tab=all&platform=${encodeURIComponent(d.label)}`;
    a.className = 'block hover:underline';

    const top = document.createElement('div');
    top.className = 'flex justify-between items-center';
    const left = document.createElement('span'); left.textContent = d.label ?? '-';
    const right = document.createElement('span'); right.textContent = d.count; right.className = 'font-semibold';
    top.append(left, right);

    const barWrap = document.createElement('div');
    barWrap.className = 'mt-2 h-2 bg-gray-800 rounded';
    const bar = document.createElement('div');
    bar.className = 'h-2 rounded bg-green-600';
    bar.style.width = `${pct}%`;
    barWrap.appendChild(bar);

    a.append(top, barWrap);
    li.appendChild(a);
    ul.appendChild(li);
  });
}

function renderFollowups(payload) {
  const tabs   = { overdue: 'fuTabOverdue', today: 'fuTabToday' };
  const fuList = document.getElementById('fuList');
  const fuEmpty= document.getElementById('fuEmpty');
  const btnO   = document.getElementById(tabs.overdue);
  const btnT   = document.getElementById(tabs.today);
  const viewAll= document.getElementById('fuViewAll');

  let current = 'overdue';

  const apply = (key) => {
    current = key;

    // tab styles
    btnO.classList.toggle('text-red-300', current === 'overdue');
    btnO.classList.toggle('text-gray-300', current !== 'overdue');
    btnT.classList.toggle('text-gray-300', current === 'overdue');

    // deep link
    if (viewAll) viewAll.href = `/admin/leads.php?tab=all&followup=${current}`;

    // rows (sorted ASC by due)
    const rows = Array.isArray(payload[current]) ? [...payload[current]] : [];
    rows.sort((a,b) => new Date(a.followup_datetime) - new Date(b.followup_datetime));

    fuList.innerHTML = '';
    if (rows.length === 0) {
      fuEmpty.textContent = '✅ No pending follow-ups';
      fuEmpty.classList.remove('hidden');
      return;
    }
    fuEmpty.classList.add('hidden');

    rows.forEach(r => {
      const li = document.createElement('li');
      li.className = 'py-2';

      // Wrapper: stacks on mobile, single row on sm+
      const wrap = document.createElement('div');
      wrap.className = 'flex flex-col gap-1 sm:flex-row sm:items-center sm:justify-between';

      // ── Row 1: avatar + broker … time
      const row1 = document.createElement('div');
      row1.className = 'flex items-center justify-between gap-2';

      const who = document.createElement('div');
      who.className = 'flex items-center gap-2 min-w-0';

      const avatar = document.createElement('span');
      avatar.className = 'inline-flex items-center justify-center h-6 w-6 rounded-full bg-blue-600 text-white text-xs';
      avatar.textContent = r.broker_initials ? r.broker_initials : '·';
      if (!r.broker_initials) avatar.classList.add('opacity-0'); // keep spacing in broker view
      who.appendChild(avatar);

      const brokerName = document.createElement('span');
      brokerName.className = 'text-xs text-gray-300 truncate';
      brokerName.textContent = r.broker_name || 'Broker';
      who.appendChild(brokerName);

      const due = document.createElement('div');
      due.className = 'text-[11px] sm:text-xs whitespace-nowrap';
      due.title = r.followup_datetime || '';
      if (current === 'overdue') {
        due.textContent = formatDueTime(r.followup_datetime);   // absolute for overdue
        due.classList.add('text-red-400');
      } else {
        due.textContent = formatRelative(r.followup_datetime);  // relative for today
        due.classList.add('text-yellow-300');
      }
      row1.append(who, due);

      // ── Row 2: #ID + status + (optional) lead name link
      const row2 = document.createElement('div');
      row2.className = 'flex items-center gap-2 text-xs text-gray-300';

      const idBadge = document.createElement('span');
      idBadge.className = 'shrink-0 font-mono text-[11px] sm:text-xs px-1.5 py-0.5 rounded bg-blue-900/40 border border-blue-700 text-blue-300';
      idBadge.textContent = `#${r.lead_id}`;
      row2.appendChild(idBadge);

      const status = document.createElement('span');
      status.className = 'px-1.5 py-0.5 rounded bg-gray-700 text-[11px] sm:text-xs text-gray-200 max-w-[10rem] truncate';
      status.textContent = r.status_name || '-';
      row2.appendChild(status);

      const leadName = (r.lead_name || '').trim();
      if (leadName) {
        const link = document.createElement('a');
        link.href = `/admin/update_lead.php?id=${r.lead_id}`;
        link.className = 'text-blue-400 hover:underline truncate max-w-[10rem] sm:max-w-none';
        link.title = `Lead #${r.lead_id} — ${leadName}`;
        link.textContent = leadName;
        row2.appendChild(link);
      }

      wrap.append(row1, row2);
      li.appendChild(wrap);
      fuList.appendChild(li);
    });
  };

  if (btnO) btnO.onclick = () => apply('overdue');
  if (btnT) btnT.onclick = () => apply('today');

  apply('overdue');
}

function renderActivity(containerId, rows) {
  const ul = document.getElementById(containerId);
  if (!ul) return;
  ul.innerHTML = '';

  rows.forEach(r => {
    const li = document.createElement('li');
    li.className = 'py-2 flex items-start justify-between gap-3';

    // Left: icon + text
    const left = document.createElement('div');
    left.className = 'flex items-start gap-3';

    const icon = document.createElement('div');
    icon.className = 'mt-0.5';
    icon.innerHTML = activityIcon(r.action);

    const text = document.createElement('div');
    text.className = 'text-gray-300';
    text.innerHTML = formatActivityText(r);

    left.append(icon, text);

    // Right: time (relative) with full timestamp on hover
    const time = document.createElement('div');
    time.className = 'text-xs text-gray-400 whitespace-nowrap';
    time.textContent = formatTime(r.created_at);
    time.title = r.created_at;

    li.append(left, time);
    ul.appendChild(li);
  });

  if (!rows || rows.length === 0) {
    const li = document.createElement('li');
    li.className = 'py-2 text-gray-400';
    li.textContent = 'No recent activity';
    ul.appendChild(li);
  }
}



function activityIcon(action) {
  switch ((action || '').toLowerCase()) {
    case 'status_update':   return '<i class="fas fa-tags text-purple-400"></i>';
    case 'comment_only':    return '<i class="fas fa-comment text-blue-400"></i>';
    case 'reassignment':
    case 'reassigned':      return '<i class="fas fa-user-exchange text-yellow-400"></i>';
    case 'unassignment':
    case 'unassigned':      return '<i class="fas fa-user-times text-red-400"></i>';
    case 'lead_created':    return '<i class="fas fa-plus-circle text-green-400"></i>';
    default:                return '<i class="fas fa-stream text-gray-400"></i>';
  }
}



function formatActivityText(r) {
  // Prefer Lead #ID, optionally append name; avoid email altogether.
  const rawName = (r.lead_name || '').trim();
  const nameSuffix = rawName ? ` — ${escapeHtml(rawName)}` : '';
  const leadLink = `<a class="text-blue-400 hover:underline" href="/admin/update_lead.php?id=${r.lead_id}">Lead #${r.lead_id}${nameSuffix}</a>`;
  const user = escapeHtml(r.user_name || 'User');

  const action = (r.action || '').toLowerCase();
  if (action === 'status_update') {
    const detail = statusDeltaPretty(r); // uses names when available
    return `<span class="text-gray-400">${user}</span> changed status ${detail} on ${leadLink}`;
  }
  if (action === 'comment_only') {
    // Truncate BEFORE escaping so entities are not cut mid-way
    const raw = String(r.comment || '');
    const trimmed = raw.length > 160 ? raw.slice(0, 160) + '…' : raw;
    const c = trimmed ? `: “${escapeHtml(trimmed)}”` : '';
    return `<span class="text-gray-400">${user}</span> added a comment${c} on ${leadLink}`;
  }
  if (action === 'reassignment' || action === 'reassigned') {
    return `<span class="text-gray-400">${user}</span> reassigned ${leadLink}`;
  }
  if (action === 'unassignment' || action === 'unassigned') {
    return `<span class="text-gray-400">${user}</span> unassigned ${leadLink}`;
  }
  if (action === 'lead_created') {
    return `<span class="text-gray-400">System</span> created ${leadLink}`;
  }
  return `<span class="text-gray-400">${user}</span> updated ${leadLink}`;
}




function statusDelta(oldId, newId) {
  if (newId && oldId && newId !== oldId) {
    return `<span class="text-gray-300">(${oldId} → ${newId})</span>`; // IDs shown; mapping to names can be added later
  }
  if (newId) return `<span class="text-gray-300">(→ ${newId})</span>`;
  return '';
}

function statusDeltaPretty(r) {
  const oldName = r.old_status_name || (r.old_status_id != null ? String(r.old_status_id) : '');
  const newName = r.new_status_name || (r.new_status_id != null ? String(r.new_status_id) : '');
  if (oldName && newName && oldName !== newName) return `<span class="text-gray-300">(${escapeHtml(oldName)} → ${escapeHtml(newName)})</span>`;
  if (!oldName && newName) return `<span class="text-gray-300">(→ ${escapeHtml(newName)})</span>`;
  if (oldName && !newName) return `<span class="text-gray-300">(${escapeHtml(oldName)} →)</span>`;
  return '';
}


function escapeHtml(s) {
  return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m]));
}

function formatRelative(ts) {
  if (!ts) return '';
  const d = new Date(ts.replace(' ', 'T') + 'Z');   // server time → treat as UTC
  const now = new Date();
  const diff = d - now;                              // future = +, past = -
  const sec = Math.round(Math.abs(diff) / 1000);

  if (sec < 60)  return diff >= 0 ? `in ${sec}s` : `${sec}s ago`;
  const min = Math.round(sec / 60);
  if (min < 60)  return diff >= 0 ? `in ${min}m` : `${min}m ago`;
  const hr  = Math.round(min / 60);
  if (hr  < 24)  return diff >= 0 ? `in ${hr}h`  : `${hr}h ago`;

  // Fallback (local absolute)
  const y = d.getFullYear();
  const m = String(d.getMonth()+1).padStart(2,'0');
  const dd= String(d.getDate()).padStart(2,'0');
  const hh= String(d.getHours()).padStart(2,'0');
  const mm= String(d.getMinutes()).padStart(2,'0');
  return `${y}-${m}-${dd} ${hh}:${mm}`;
}

// ADD this helper (e.g., just above formatTime)
function formatDueTime(ts) {
  if (!ts) return '-';
  // server "YYYY-MM-DD HH:MM:SS" → treat as UTC for consistency
  const d = new Date(ts.replace(' ', 'T') + 'Z');
  const y  = d.getFullYear();
  const m  = String(d.getMonth() + 1).padStart(2, '0');
  const dd = String(d.getDate()).padStart(2, '0');
  const hh = String(d.getHours()).padStart(2, '0');
  const mm = String(d.getMinutes()).padStart(2, '0');
  return `${y}-${m}-${dd} ${hh}:${mm}`;
}


function formatTime(ts) {
  // ts: "YYYY-MM-DD HH:MM:SS" (server time). Convert safely to local.
  if (!ts) return '';
  const iso = ts.replace(' ', 'T') + 'Z'; // treat server as UTC for consistency
  const d = new Date(iso);
  const now = new Date();
  const diffMs = now - d;
  const sec = Math.floor(diffMs / 1000);
  if (sec < 60) return `${sec}s ago`;
  const min = Math.floor(sec / 60);
  if (min < 60) return `${min}m ago`;
  const hr = Math.floor(min / 60);
  if (hr < 24) return `${hr}h ago`;
  const day = Math.floor(hr / 24);
  if (day === 1) return 'yesterday';
  if (day < 7) return `${day}d ago`;
  // Fallback: concise absolute
  const y = d.getFullYear();
  const m = String(d.getMonth() + 1).padStart(2, '0');
  const dd = String(d.getDate()).padStart(2, '0');
  const hh = String(d.getHours()).padStart(2, '0');
  const mm = String(d.getMinutes()).padStart(2, '0');
  return `${y}-${m}-${dd} ${hh}:${mm}`;
}

