// assets/js/brokers.js
// Modular broker script for Knight Castle CRM admin

let currentEditId = 0; // Tracks which user is being edited

// ✅ Load all brokers from API and render into table/card
export async function loadBrokers() {
  try {
    const res = await fetch('/api/brokers_api.php?action=fetch', {
      method: 'GET',
      headers: { 'Content-Type': 'application/json' }
    });

    const data = await res.json();
    if (data.success && Array.isArray(data.data)) {
      renderBrokerTable(data.data); // Auto-switches to table or card
    } else {
      showToast('Failed to load users', 'error');
    }
  } catch (error) {
    console.error("❌ Error loading brokers:", error);
    showToast('Network error while loading users', 'error');
  }
}

// ✅ Renders brokers in desktop <table> or mobile <div> card view
function renderBrokerTable(users) {
  const tableBody = document.getElementById('brokerTableBody');
  const cardContainer = document.getElementById('brokerCardContainer');

  if (!tableBody || !cardContainer) return;

  // Clear both views before injecting new data
  tableBody.innerHTML = '';
  cardContainer.innerHTML = '';

  const isDesktop = window.innerWidth >= 1024; // Tailwind lg breakpoint

  users.forEach(user => {
    const isSuperAdmin = user.id === 1;

    // 🧠 Prepare conditional delete button
    const deleteButton = isSuperAdmin ? '' : `
      <button class="deleteBtn bg-red-600 hover:bg-red-700 text-white px-3 py-1 rounded" data-id="${user.id}">Delete</button>
    `;

    if (isDesktop) {
      // 🖥 Render table row
      const row = document.createElement('tr');
      row.className = 'border-b border-gray-700 hover:bg-gray-800';
      row.innerHTML = `
        <td class="px-4 py-2">${user.id}</td>
        <td class="px-4 py-2">${user.name}</td>
        <td class="px-4 py-2">${user.company_name}</td>
        <td class="px-4 py-2">${user.email}</td>
        <td class="px-4 py-2">${user.phone}</td>
        <td class="px-4 py-2 capitalize">${user.role}</td>
        <td class="px-4 py-2 capitalize">${user.status}</td>
        <td class="px-4 py-2 text-sm">
          <button class="editBtn bg-yellow-600 hover:bg-yellow-700 text-white px-3 py-1 rounded mr-2" data-id="${user.id}">Edit</button>
          ${deleteButton}
        </td>
      `;
      tableBody.appendChild(row);

      row.querySelector('.editBtn').addEventListener('click', () => populateForm(user));
      if (!isSuperAdmin) {
        row.querySelector('.deleteBtn').addEventListener('click', () => confirmDelete(user.id));
      }
    } else {
      // 📱 Render mobile card
      const card = document.createElement('div');
      card.className = 'bg-gray-800 rounded-lg shadow p-4 text-sm space-y-2';
      card.innerHTML = `
        <div><strong class="text-gray-400">Name:</strong> ${user.name}</div>
        <div><strong class="text-gray-400">Company:</strong> ${user.company_name}</div>
        <div><strong class="text-gray-400">Email:</strong> ${user.email}</div>
        <div><strong class="text-gray-400">Phone:</strong> ${user.phone}</div>
        <div><strong class="text-gray-400">Role:</strong> ${user.role}</div>
        <div><strong class="text-gray-400">Status:</strong> ${user.status}</div>
        <div class="flex gap-2 pt-2">
          <button class="editBtn bg-yellow-600 hover:bg-yellow-700 text-white px-4 py-1 rounded" data-id="${user.id}">Edit</button>
          ${isSuperAdmin ? '' : `
            <button class="deleteBtn bg-red-600 hover:bg-red-700 text-white px-4 py-1 rounded" data-id="${user.id}">Delete</button>
          `}
        </div>
      `;
      cardContainer.appendChild(card);

      card.querySelector('.editBtn').addEventListener('click', () => populateForm(user));
      if (!isSuperAdmin) {
        card.querySelector('.deleteBtn').addEventListener('click', () => confirmDelete(user.id));
      }
    }
  });
}

// ✅ Populate form fields for editing
function populateForm(user) {
  currentEditId = user.id;

  document.getElementById('name').value = user.name;
  document.getElementById('company_name').value = user.company_name;
  document.getElementById('email').value = user.email;
  document.getElementById('phone').value = user.phone;
  document.getElementById('role').value = user.role;
  document.getElementById('status').value = user.status;
  document.getElementById('password').value = '';

  // 🔒 Prevent editing role/email/status for Super Admin
  const isSuperAdmin = user.id === 1;
  document.getElementById('email').readOnly = isSuperAdmin;
  document.getElementById('role').disabled = isSuperAdmin;
  document.getElementById('status').disabled = isSuperAdmin;
}

// ✅ Delete user with confirmation
async function confirmDelete(id) {
  if (!confirm("Are you sure you want to delete this user?")) return;

  const res = await fetch('/api/brokers_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ action: 'delete', id })
  });

  const result = await res.json();
  if (result.success) {
    loadBrokers(); // Refresh list
    showToast('User deleted successfully', 'success');
  } else {
    showToast(result.message || 'Failed to delete user', 'error');
  }
}

// ✅ Save (create or update) user
document.getElementById('saveBtn').addEventListener('click', async () => {
  const payload = {
    action: 'create_or_update',
    id: currentEditId,
    name: document.getElementById('name').value.trim(),
    company_name: document.getElementById('company_name').value.trim(),
    email: document.getElementById('email').value.trim(),
    phone: document.getElementById('phone').value.trim(),
    role: document.getElementById('role').value,
    status: document.getElementById('status').value,
    password: document.getElementById('password').value.trim()
  };

  // 🔒 Force Super Admin role protection on client side
  if (payload.id === 1) {
    payload.role = 'admin';
    payload.status = 'active';
  }

  const res = await fetch('/api/brokers_api.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(payload)
  });

  const result = await res.json();
  if (result.success) {
    showToast('User saved successfully', 'success');
    resetForm();
    loadBrokers(); // Reload updated list
  } else {
    showToast(result.message || 'Failed to save user', 'error');
  }
});

// ✅ Reset form fields and re-enable all controls
document.getElementById('resetBtn').addEventListener('click', resetForm);

function resetForm() {
  currentEditId = 0;
  document.getElementById('brokerForm').reset();
  document.getElementById('password').value = '';
  document.getElementById('email').readOnly = false;
  document.getElementById('role').disabled = false;
  document.getElementById('status').disabled = false;
}

// ✅ Toast Notification Utility
function showToast(message, type = 'info') {
  const toast = document.createElement('div');
  toast.textContent = message;
  toast.className = `fixed top-4 right-4 px-4 py-2 rounded text-white z-50 transition 
    ${type === 'error' ? 'bg-red-600' : type === 'success' ? 'bg-green-600' : 'bg-gray-700'}`;
  document.body.appendChild(toast);
  setTimeout(() => toast.remove(), 3500);
}

// ✅ Load data after DOM is ready
window.addEventListener('DOMContentLoaded', () => {
  loadBrokers();
});
