<?php
// api/unassign_broker.php

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json');

require_once '../config.php';
require_once '../lib/auth.php';

// 🔒 Only admins allowed
if (!isLoggedIn() || $_SESSION['role'] !== 'admin') {
  http_response_code(403);
  echo json_encode(['error' => 'Unauthorized']);
  exit;
}

// 📥 Parse and validate JSON input
$data = json_decode(file_get_contents("php://input"), true);
$leadId   = (int)($data['lead_id'] ?? 0);
$brokerId = (int)($data['broker_id'] ?? 0);

if (!$leadId || !$brokerId) {
  http_response_code(400);
  echo json_encode(['error' => 'Missing required data']);
  exit;
}

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['error' => 'DB connection failed']);
  exit;
}

// ✅ Verify if broker is assigned to lead
$check = $conn->prepare("SELECT 1 FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
$check->bind_param("ii", $leadId, $brokerId);
$check->execute();
$check->store_result();

if ($check->num_rows === 0) {
  $check->close();
  echo json_encode(['error' => 'Broker not assigned to this lead']);
  exit;
}
$check->close();

// ❌ Remove from lead_broker_map
$delMap = $conn->prepare("DELETE FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
$delMap->bind_param("ii", $leadId, $brokerId);
$delMap->execute();
$delMap->close();

// 🧹 Soft delete from lead_broker_status
$softDelete = $conn->prepare("UPDATE lead_broker_status SET deleted = 1 WHERE lead_id = ? AND broker_id = ?");
$softDelete->bind_param("ii", $leadId, $brokerId);
$softDelete->execute();
$softDelete->close();

// 📝 Log the action in lead_logs
$adminId = $_SESSION['user_id'];
$comment = "Admin unassigned broker ID $brokerId from lead ID $leadId.";

$log = $conn->prepare("
  INSERT INTO lead_logs 
  (lead_id, action, old_broker_id, new_broker_id, comment, changed_by, role, created_at) 
  VALUES (?, 'unassignment', ?, NULL, ?, ?, 'admin', NOW())
");
$log->bind_param("iisi", $leadId, $brokerId, $comment, $adminId);

$log->execute();
$log->close();

$conn->close();
echo json_encode(['success' => true]);
