<?php
// ✅ api/sheets_api.php
// JSON-based handler for managing monitored_sheets
// Supports: list, create_or_update, get, delete, sync, validate_sheet
// Uses JSON only (⛔ no FormData)

if (session_status() === PHP_SESSION_NONE) session_start();
require_once '../config.php';
require_once '../lib/auth.php';

header('Content-Type: application/json');

// 🔐 Admin-only access
if (!isLoggedIn() || $_SESSION['role'] !== 'admin') {
  http_response_code(403);
  echo json_encode(['success' => false, 'message' => 'Unauthorized']);
  exit;
}

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['success' => false, 'message' => 'Database connection error']);
  exit;
}

// ✅ Parse JSON input
$input = json_decode(file_get_contents("php://input"), true);
$action = $input['action'] ?? '';

if (!$action) {
  echo json_encode(['success' => false, 'message' => 'Missing action']);
  exit;
}

// 🔁 Sync to monitored_sheets.json
function sync_to_json($conn) {
  $result = $conn->query("SELECT sheet_id, tab_name, last_synced_row, notes FROM monitored_sheets");
  $data = ['sources' => []];
  while ($row = $result->fetch_assoc()) {
    $data['sources'][] = $row;
  }
  file_put_contents(__DIR__ . '/../google_api/monitored_sheets.json', json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

// ▶️ Log one admin action into monitored_sheet_logs
function log_sheet_action(mysqli $conn, int $sheet_db_id, string $action, $old = null, $new = null): void {
  $oldV = (is_array($old) || is_object($old)) ? json_encode($old, JSON_UNESCAPED_UNICODE) : (string)$old;
  $newV = (is_array($new) || is_object($new)) ? json_encode($new, JSON_UNESCAPED_UNICODE) : (string)$new;
  $performed_by = (int)($_SESSION['user_id'] ?? 0);

  $stmt = $conn->prepare("
    INSERT INTO monitored_sheet_logs (sheet_id, action, old_value, new_value, performed_by, created_at)
    VALUES (?, ?, ?, ?, ?, NOW())
  ");
  $stmt->bind_param("isssi", $sheet_db_id, $action, $oldV, $newV, $performed_by);
  $stmt->execute();
  $stmt->close();
}


switch ($action) {

  case 'list':
    // 🧾 Return all monitored sheets
    $sheets = [];
    $res = $conn->query("SELECT * FROM monitored_sheets ORDER BY id DESC");
    while ($row = $res->fetch_assoc()) $sheets[] = $row;
    echo json_encode(['success' => true, 'sheets' => $sheets]);
    break;

  case 'create_or_update':
  // 💾 Save or update a monitored sheet
  $id = (int) ($input['id'] ?? 0);
  $sheet_id = trim($input['sheet_id'] ?? '');
  $tab_name = trim($input['tab_name'] ?? '');
  $notes = trim($input['notes'] ?? '');
  $last_synced_row = (int) ($input['last_synced_row'] ?? 0);

  if (!$sheet_id || !$tab_name) {
    echo json_encode(['success' => false, 'message' => 'Sheet ID and Tab Name are required']);
    break;
  }

  // Snapshot for logging (old values if updating)
  $oldRow = null;
  if ($id > 0) {
    $get = $conn->prepare("SELECT * FROM monitored_sheets WHERE id = ? LIMIT 1");
    $get->bind_param("i", $id);
    $get->execute();
    $res = $get->get_result();
    $oldRow = $res->fetch_assoc();
    $get->close();
  }

  if ($id > 0) {
    $stmt = $conn->prepare("UPDATE monitored_sheets SET sheet_id = ?, tab_name = ?, notes = ?, last_synced_row = ? WHERE id = ?");
    $stmt->bind_param("sssii", $sheet_id, $tab_name, $notes, $last_synced_row, $id);
  } else {
    $stmt = $conn->prepare("INSERT INTO monitored_sheets (sheet_id, tab_name, notes, last_synced_row) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("sssi", $sheet_id, $tab_name, $notes, $last_synced_row);
  }

  $success = $stmt->execute();
  $newId = ($id > 0) ? $id : (int)$conn->insert_id;
  $stmt->close();

  if ($success) {
    sync_to_json($conn);

    // 📝 Log admin action
    $newRow = [
      'sheet_id' => $sheet_id,
      'tab_name' => $tab_name,
      'notes' => $notes,
      'last_synced_row' => $last_synced_row
    ];
    log_sheet_action($conn, $newId, ($id > 0 ? 'update' : 'create'), $oldRow, $newRow);
  }

  echo json_encode(['success' => $success, 'id' => $newId]);
  break;


  case 'get':
    // ✏️ Fetch one sheet by ID
    $id = (int) ($input['id'] ?? 0);
    $stmt = $conn->prepare("SELECT * FROM monitored_sheets WHERE id = ? LIMIT 1");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $res = $stmt->get_result();
    $sheet = $res->fetch_assoc();
    echo json_encode(['success' => true, 'sheet' => $sheet]);
    break;

    case 'delete':
    // ❌ Delete monitored sheet by ID, with optional cascade to its leads
    $id = (int) ($input['id'] ?? 0);
    $cascade = !empty($input['cascade_delete_leads']); // boolean flag from UI

    // 1) Look up tab_name for this sheet
    $getTab = $conn->prepare("SELECT tab_name FROM monitored_sheets WHERE id = ? LIMIT 1");
    $getTab->bind_param("i", $id); // id is int
    $getTab->execute();
    $tabRes = $getTab->get_result();
    $tabRow = $tabRes->fetch_assoc();
    $tab_name = $tabRow['tab_name'] ?? null;

    if (!$tab_name) {
      echo json_encode(['success' => false, 'message' => 'Sheet not found']);
      break;
    }

    $deletedLeads = 0;

    if ($cascade) {
      // 2) Cascade delete: remove related leads and mappings for this tab_name
      $conn->begin_transaction();

      // 2a) Collect lead IDs for this tab_name
      $ids = [];
      $sel = $conn->prepare("SELECT id FROM leads WHERE tab_name = ?");
      $sel->bind_param("s", $tab_name); // tab_name is string
      $sel->execute();
      $r = $sel->get_result();
      while ($row = $r->fetch_assoc()) { $ids[] = (int)$row['id']; }
      $deletedLeads = count($ids);

      // 2b) Delete per-lead rows in dependent tables; loop for safe bind_param usage
      if ($deletedLeads > 0) {
        // lead_logs
        $delLogs = $conn->prepare("DELETE FROM lead_logs WHERE lead_id = ?");
        // lead_broker_status
        $delLbs = $conn->prepare("DELETE FROM lead_broker_status WHERE lead_id = ?");
        // lead_broker_map
        $delLbm = $conn->prepare("DELETE FROM lead_broker_map WHERE lead_id = ?");
        // leads
        $delLead = $conn->prepare("DELETE FROM leads WHERE id = ?");

        foreach ($ids as $lid) {
          $delLogs->bind_param("i", $lid); $delLogs->execute();   // lead_id is int
          $delLbs->bind_param("i", $lid);  $delLbs->execute();    // lead_id is int
          $delLbm->bind_param("i", $lid);  $delLbm->execute();    // lead_id is int
          $delLead->bind_param("i", $lid); $delLead->execute();   // id is int
        }
      }

      // 2c) Finally delete the monitored_sheets row
      $stmt = $conn->prepare("DELETE FROM monitored_sheets WHERE id = ?");
      $stmt->bind_param("i", $id); // id is int
      $ok = $stmt->execute();

      if ($ok) {
        $conn->commit();
        sync_to_json($conn); // keep JSON in sync after structural changes
        echo json_encode(['success' => true, 'deleted_leads' => $deletedLeads]);
      } else {
        $conn->rollback();
        echo json_encode(['success' => false, 'message' => 'Failed to delete sheet']);
      }
    } else {
      // 3) Keep leads; only remove the sheet row and resync
      $stmt = $conn->prepare("DELETE FROM monitored_sheets WHERE id = ?");
      $stmt->bind_param("i", $id); // id is int
      $success = $stmt->execute();
      if ($success) sync_to_json($conn);
      
      // 📝 Log admin deletion with snapshot of the removed row
$deletedRow = ['tab_name' => $tab_name, 'id' => $id];
log_sheet_action($conn, $id, 'delete', $deletedRow, null);

      echo json_encode(['success' => $success, 'deleted_leads' => 0]);
    }
    break;


  case 'sync':
    // 🔄 Regenerate monitored_sheets.json
    sync_to_json($conn);
    // optional: log a sync event for each sheet (or one generic log)
// Simple generic log for sheet_id 0 (system)
$conn->query("
  INSERT INTO monitored_sheet_logs (sheet_id, action, old_value, new_value, performed_by, created_at)
  VALUES (0, 'sync', NULL, NULL, ".(int)($_SESSION['user_id'] ?? 0).", NOW())
");

    echo json_encode(['success' => true]);
    break;

case 'validate_sheet':
  // ✅ Check if Sheet ID and Tab exist
  $sheet_id = trim($input['sheet_id'] ?? '');
  $tab_name = trim($input['tab_name'] ?? '');

  if (!$sheet_id || !$tab_name) {
    echo json_encode(['success' => false, 'message' => 'Sheet ID and Tab Name required']);
    exit;
  }

  try {
    require_once __DIR__ . '/../google_api/sheets_client.php';

    $client = getClient();
    $service = new Google_Service_Sheets($client);
    $spreadsheet = $service->spreadsheets->get($sheet_id);
    $tabs = array_map(fn($s) => $s->getProperties()->getTitle(), $spreadsheet->getSheets());

    $valid = in_array($tab_name, $tabs);
    echo json_encode([
      'success' => $valid,
      'message' => $valid ? 'Sheet and tab verified' : 'Tab not found in provided sheet'
    ]);
  } catch (Google_Service_Exception $e) {
    error_log("Sheet validation failed: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Invalid Sheet ID or access denied']);
  } catch (Exception $e) {
    error_log("Validation error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Unexpected error during validation']);
  }
  break;

case 'logs':
  // 📜 Return recent activity for a sheet (admin-only page)
  $sheet_db_id = (int)($input['id'] ?? 0);
  $limit = max(1, min(100, (int)($input['limit'] ?? 50)));

    $stmt = $conn->prepare("
    SELECT
      l.id, l.action, l.old_value, l.new_value, l.performed_by, l.created_at,
      CASE
        WHEN l.performed_by = 0 THEN 'System'
        ELSE COALESCE(b.name, b.email, CONCAT('User #', l.performed_by))
      END AS performed_by_name
    FROM monitored_sheet_logs l
    LEFT JOIN brokers b ON b.id = l.performed_by
    WHERE l.sheet_id = ?
    ORDER BY l.id DESC
    LIMIT ?
  ");

  $stmt->bind_param("ii", $sheet_db_id, $limit);
  $stmt->execute();
  $res = $stmt->get_result();

  $rows = [];
  while ($r = $res->fetch_assoc()) {
    $rows[] = [
      'id' => (int)$r['id'],
      'action' => $r['action'],
      'old' => $r['old_value'],
      'new' => $r['new_value'],
      'by'  => $r['performed_by_name'] ?: ((int)$r['performed_by'] === 0 ? 'System' : 'User #' . (int)$r['performed_by']),

      'at'  => $r['created_at']
    ];
  }
  $stmt->close();

  echo json_encode(['success' => true, 'logs' => $rows]);
  break;
  
  case 'logs_all':
  // consolidated timeline across all sheets
  $limit = max(1, min(200, (int)($input['limit'] ?? 100)));
    $stmt = $conn->prepare("
    SELECT
      l.id, l.action, l.old_value, l.new_value, l.performed_by, l.created_at,
      CASE
        WHEN l.performed_by = 0 THEN 'System'
        ELSE COALESCE(b.name, b.email, CONCAT('User #', l.performed_by))
      END AS performed_by_name,
      ms.tab_name,
      ms.id AS sheet_db_id
    FROM monitored_sheet_logs l
    JOIN monitored_sheets ms ON ms.id = l.sheet_id
    LEFT JOIN brokers b ON b.id = l.performed_by
    ORDER BY l.id DESC
    LIMIT ?
  ");

  $stmt->bind_param("i", $limit);
  $stmt->execute();
  $res = $stmt->get_result();

  $rows = [];
  while ($r = $res->fetch_assoc()) {
    $rows[] = [
      'id'   => (int)$r['id'],
      'sheet_db_id' => (int)$r['sheet_db_id'],
      'tab_name'    => $r['tab_name'],
      'action'      => $r['action'],
      'old'         => $r['old_value'],
      'new'         => $r['new_value'],
      'by'          => $r['performed_by_name'],

      'at'          => $r['created_at']
    ];
  }
  $stmt->close();
  echo json_encode(['success' => true, 'logs' => $rows]);
  break;

case 'import_now':
  // Run the same importer used by cron, synchronously.
  // It prints nothing and exits cleanly.
  try {
    require_once __DIR__ . '/../cron/import_leads.php';
    // If we got here without fatal, treat as success.
    echo json_encode(['success' => true]);
  } catch (Throwable $e) {
    error_log('import_now failed: '.$e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Importer error']);
  }
  break;


  default:
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    break;
}
?>
