<?php
// api/notifications.php
// JSON-only notifications API: count, list, mark_one_read, mark_all_read
// Lead identity = leads.id everywhere. Notifications truncated to last 10 days.

session_start();
header('Content-Type: application/json');

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/auth.php';

// --- Auth Check ---
if (!isset($_SESSION['user_id'])) {
  echo json_encode(["success" => false, "error" => "Unauthorized"]);
  exit;
}

$userId   = (int)$_SESSION['user_id'];
$userRole = $_SESSION['role'] ?? 'broker';

// --- Parse Input ---
$input  = json_decode(file_get_contents('php://input'), true) ?: [];
$mode   = $input['mode']   ?? null;
$action = $input['action'] ?? null;

// --- DB Connect ---
$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_errno) {
  echo json_encode(["success" => false, "error" => "DB connect error"]);
  exit;
}

// ---------------- Housekeeping ----------------
$cutoff = date('Y-m-d H:i:s', strtotime('-10 days'));
$purge = $db->prepare("DELETE FROM user_notifications_read WHERE created_at < ?");
$purge->bind_param("s", $cutoff);
$purge->execute();
$purge->close();

/* ============================================================
   MODE: COUNT
   ------------------------------------------------------------
   Admin:
     - Sees all unread notifications across all leads.
   Broker:
     - Sees assignment/reassignment/unassignment events where
       they are directly involved (new_broker_id / old_broker_id).
     - Sees status/comment/followup only for leads where they
       are the actor (changed_by = broker_id).
============================================================ */
if ($mode === 'count') {
  $sql = "
    SELECT COUNT(*) AS unread_count
    FROM lead_logs ll
    INNER JOIN leads l ON ll.lead_id = l.id
    LEFT JOIN user_notifications_read unr
      ON unr.user_id = ?
     AND unr.notification_type = ll.action
     AND unr.related_id = ll.lead_id
  ";

  if ($userRole === 'broker') {
    $sql .= "
      LEFT JOIN lead_broker_map lbm
        ON lbm.lead_id = ll.lead_id
       AND lbm.broker_id = ?
      WHERE unr.id IS NULL
        AND ll.action IS NOT NULL
        AND ll.created_at >= ?
        AND (
          (ll.action IN ('assignment','unassignment','reassignment')
            AND (ll.new_broker_id = ? OR ll.old_broker_id = ?))
          OR
          (ll.action IN ('status_update','comment_only','followup')
            AND ll.changed_by = ?)
        )
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param(
      "iisiii",
      $userId,  // unr.user_id
      $userId,  // lbm.broker_id
      $cutoff,  // cutoff date
      $userId,  // new_broker_id
      $userId,  // old_broker_id
      $userId   // changed_by
    );
  } else {
    $sql .= " WHERE unr.id IS NULL AND ll.action IS NOT NULL AND ll.created_at >= ?";
    $stmt = $db->prepare($sql);
    $stmt->bind_param("is", $userId, $cutoff);
  }

  $stmt->execute();
  $res = $stmt->get_result();
  $cnt = (int)($res->fetch_assoc()['unread_count'] ?? 0);
  $stmt->close();

  echo json_encode(["success" => true, "unread_count" => $cnt]);
  exit;
}

/* ============================================================
   ACTION: MARK ONE READ
   ------------------------------------------------------------
   Marks a single notification as read for this user.
============================================================ */
if ($action === 'mark_one_read') {
  $type      = $input['type'] ?? null;
  $relatedId = isset($input['related_id']) ? (int)$input['related_id'] : 0;

  if (!$type || $relatedId <= 0) {
    echo json_encode(["success" => false, "error" => "Missing fields"]);
    exit;
  }

  $stmt = $db->prepare("
    INSERT IGNORE INTO user_notifications_read (user_id, notification_type, related_id)
    VALUES (?, ?, ?)
  ");
  $stmt->bind_param("isi", $userId, $type, $relatedId);
  $stmt->execute();
  $stmt->close();

  echo json_encode(["success" => true]);
  exit;
}

/* ============================================================
   ACTION: MARK ALL READ
   ------------------------------------------------------------
   Marks all visible notifications (within scope) as read.
============================================================ */
if ($action === 'mark_all_read') {
  $sql = "
    INSERT IGNORE INTO user_notifications_read (user_id, notification_type, related_id)
    SELECT ?, ll.action, ll.lead_id
      FROM lead_logs ll
      INNER JOIN leads l ON ll.lead_id = l.id
      LEFT JOIN user_notifications_read unr
        ON unr.user_id = ?
       AND unr.notification_type = ll.action
       AND unr.related_id = ll.lead_id
  ";

  if ($userRole === 'broker') {
    $sql .= "
      LEFT JOIN lead_broker_map lbm
        ON lbm.lead_id = ll.lead_id
       AND lbm.broker_id = ?
      WHERE unr.id IS NULL
        AND ll.action IS NOT NULL
        AND ll.created_at >= ?
        AND (
          (ll.action IN ('assignment','unassignment','reassignment')
            AND (ll.new_broker_id = ? OR ll.old_broker_id = ?))
          OR
          (ll.action IN ('status_update','comment_only','followup')
            AND ll.changed_by = ?)
        )
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param(
      "iiisiii",
      $userId,  // SELECT ? (for insert)
      $userId,  // unr.user_id
      $userId,  // lbm.broker_id
      $cutoff,  // created_at cutoff
      $userId,  // new_broker_id
      $userId,  // old_broker_id
      $userId   // changed_by
    );
  } else {
    $sql .= "
      WHERE unr.id IS NULL
        AND ll.action IS NOT NULL
        AND ll.created_at >= ?
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param("iis", $userId, $userId, $cutoff);
  }

  $stmt->execute();
  $stmt->close();

  echo json_encode(["success" => true]);
  exit;
}

/* ============================================================
   MODE: LIST
   ------------------------------------------------------------
   Admin:
     - Sees all notifications from last 10 days.
   Broker:
     - Sees only:
         a) assignment/reassignment/unassignment if directly involved
         b) status/comment/followup if they performed it (changed_by)
============================================================ */
if ($mode === 'list') {
  $limit  = max(1, (int)($input['limit']  ?? 20));
  $offset = max(0, (int)($input['offset'] ?? 0));

  $sql = "
    SELECT
      ll.action  AS type,
      ll.lead_id AS related_id,
      ll.comment,
      ll.followup_datetime,
      ll.created_at AS time,
      u.name     AS actor_name,
      u.role     AS actor_role,
      CASE WHEN unr.id IS NULL THEN 0 ELSE 1 END AS is_read
    FROM lead_logs ll
    INNER JOIN leads l ON ll.lead_id = l.id
    INNER JOIN users u ON ll.changed_by = u.id
    LEFT JOIN user_notifications_read unr
      ON unr.user_id = ?
     AND unr.notification_type = ll.action
     AND unr.related_id = ll.lead_id
  ";

  if ($userRole === 'broker') {
    $sql .= "
      LEFT JOIN lead_broker_map lbm
        ON lbm.lead_id = ll.lead_id
       AND lbm.broker_id = ?
      WHERE ll.created_at >= ?
        AND ll.action IS NOT NULL
        AND (
          (ll.action IN ('assignment','unassignment','reassignment')
            AND (ll.new_broker_id = ? OR ll.old_broker_id = ?))
          OR
          (ll.action IN ('status_update','comment_only','followup')
            AND ll.changed_by = ?)
        )
      ORDER BY ll.created_at DESC
      LIMIT ? OFFSET ?
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param(
      "iisiiiii",
      $userId,  // unr.user_id
      $userId,  // lbm.broker_id
      $cutoff,
      $userId,  // new_broker_id
      $userId,  // old_broker_id
      $userId,  // changed_by
      $limit,
      $offset
    );
  } else {
    $sql .= "
      WHERE ll.created_at >= ? AND ll.action IS NOT NULL
      ORDER BY ll.created_at DESC
      LIMIT ? OFFSET ?
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param("isii", $userId, $cutoff, $limit, $offset);
  }

  $stmt->execute();
  $res = $stmt->get_result();

  $notifications = [];
  while ($row = $res->fetch_assoc()) {
    $notifications[] = [
      "type"        => $row['type'],
      "related_id"  => (int)$row['related_id'],
      "message"     => buildNotificationMessage($row),
      "time"        => $row['time'],
      "is_read"     => (bool)$row['is_read'],
      "actor_name"  => $row['actor_name'],
      "actor_role"  => $row['actor_role']
    ];
  }
  $stmt->close();

  // Unread count (same scoped logic)
  $countSql = "
    SELECT COUNT(*) AS unread_count
    FROM lead_logs ll
    INNER JOIN leads l ON ll.lead_id = l.id
    LEFT JOIN user_notifications_read unr
      ON unr.user_id = ?
     AND unr.notification_type = ll.action
     AND unr.related_id = ll.lead_id
  ";

  if ($userRole === 'broker') {
    $countSql .= "
      LEFT JOIN lead_broker_map lbm
        ON lbm.lead_id = ll.lead_id
       AND lbm.broker_id = ?
      WHERE unr.id IS NULL
        AND ll.action IS NOT NULL
        AND ll.created_at >= ?
        AND (
          (ll.action IN ('assignment','unassignment','reassignment')
            AND (ll.new_broker_id = ? OR ll.old_broker_id = ?))
          OR
          (ll.action IN ('status_update','comment_only','followup')
            AND ll.changed_by = ?)
        )
    ";
    $cst = $db->prepare($countSql);
    $cst->bind_param(
      "iisiii",
      $userId,  // unr.user_id
      $userId,  // lbm.broker_id
      $cutoff,
      $userId,  // new_broker_id
      $userId,  // old_broker_id
      $userId   // changed_by
    );
  } else {
    $countSql .= " WHERE unr.id IS NULL AND ll.action IS NOT NULL AND ll.created_at >= ?";
    $cst = $db->prepare($countSql);
    $cst->bind_param("is", $userId, $cutoff);
  }

  $cst->execute();
  $cres = $cst->get_result();
  $unread = (int)($cres->fetch_assoc()['unread_count'] ?? 0);
  $cst->close();

  echo json_encode([
    "success"       => true,
    "notifications" => $notifications,
    "unread_count"  => $unread,
    "has_more"      => (count($notifications) === $limit)
  ]);
  exit;
}

/* ============================================================
   DEFAULT HANDLER
============================================================ */
echo json_encode(["success" => false, "error" => "Invalid request"]);

/* ============================================================
   Helper: Build Notification Message
============================================================ */
function buildNotificationMessage(array $row) {
  $id   = (int)$row['related_id'];
  $user = $row['actor_name'] ?? 'Unknown';
  $role = $row['actor_role'] ?? '';

  switch ($row['type']) {
    case 'assignment':    return "Lead {$id} assigned by {$user} ({$role})";
    case 'unassignment':  return "Lead {$id} unassigned by {$user} ({$role})";
    case 'reassignment':  return "Lead {$id} reassigned by {$user} ({$role})";
    case 'status_update': return "Lead {$id} status updated by {$user} ({$role})";
    case 'comment_only':  return "New comment on lead {$id} by {$user} ({$role})";
    case 'followup':      return "Follow-up set for lead {$id} by {$user} ({$role})";
    case 'new_lead':      return "New lead {$id} synced (unassigned)";
    default:              return "Activity on lead {$id} by {$user} ({$role})";
  }
}
