<?php
// api/log_lead_action.php
// Handles: status_update, followup_update, comment_only, reassignment, unassignment

session_start();
header('Content-Type: application/json');

// Safe JSON error responder
function jsonError($message, $code = 500) {
  http_response_code($code);
  echo json_encode(['error' => $message]);
  exit;
}

require_once '../config.php';
require_once '../lib/auth.php';

// Clean all output buffers to prevent stray whitespace
while (ob_get_level()) {
    ob_end_clean();
}

// Auth
if (!isset($_SESSION['user_id'], $_SESSION['role'])) {
  jsonError('Unauthorized', 401);
}

$changed_by = (int) $_SESSION['user_id'];
$role       = $_SESSION['role'];

// Parse input
$input          = json_decode(file_get_contents('php://input'), true);
$lead_id        = (int)($input['lead_id'] ?? 0);
$action         = trim($input['action'] ?? '');
$old_status_id  = isset($input['old_status_id']) ? (int)$input['old_status_id'] : null;
$new_status_id  = isset($input['new_status_id']) ? (int)$input['new_status_id'] : null;
$new_broker_id  = isset($input['new_broker_id']) ? (int)$input['new_broker_id'] : null;
$old_broker_id = isset($input['old_broker_id']) ? (int)$input['old_broker_id'] : null; // broker being replaced, if any

$commentRaw     = $input['comment'] ?? '';
$comment        = is_string($commentRaw) ? trim($commentRaw) : '';
$followup_dt_raw = trim($input['followup_datetime'] ?? '');
$followup_dt = $followup_dt_raw ? str_replace('T', ' ', $followup_dt_raw) : null; // normalize HTML5 datetime


$broker_id      = isset($input['broker_id']) ? (int)$input['broker_id'] : $changed_by; // for unassignment/status updates

// Identify the broker whose lead data was affected (important for admin updates)
$affected_broker_id = $broker_id > 0 ? $broker_id : $changed_by;


if ($comment === '0') $comment = '';
if (!$lead_id || !$action)  jsonError('Missing lead_id or action', 400);
if (in_array($action, ['reassignment', 'unassignment']) && $role !== 'admin') {
  jsonError('Only admins can perform this action', 403);
}

// DB
$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) jsonError('DB connection error', 500);

// Follow-up rules
$requires_followup_statuses = [
  'Contacted', 'Engaged', 'Interested', 'Qualified',
  'Site Visit Scheduled', 'Site Visit Done', 'Booking in Progress'
];

if ($action === 'status_update' && $new_status_id) {
  $stmt = $conn->prepare("SELECT name FROM lead_statuses WHERE id = ?");
  if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
  $stmt->bind_param("i", $new_status_id);
  $stmt->execute();
  $stmt->bind_result($status_name);
  $stmt->fetch();
  $stmt->close();

  if (in_array($status_name, $requires_followup_statuses) && !$followup_dt) {
    jsonError("Follow-up required for status: $status_name", 400);
  }
}

// Route
switch ($action) {
  case 'status_update':
    // upsert broker status row
    $sql = "INSERT INTO lead_broker_status (lead_id, broker_id, status_id, followup_datetime)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE status_id = VALUES(status_id),
                                    followup_datetime = VALUES(followup_datetime)";
    $stmt = $conn->prepare($sql);
    if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt->bind_param("iiis", $lead_id, $broker_id, $new_status_id, $followup_dt);
    $stmt->execute();
    $stmt->close();

    // log
    $stmt = $conn->prepare("
      INSERT INTO lead_logs
  (lead_id, action, old_status_id, new_status_id, comment, followup_datetime, changed_by, role, affected_broker_id, created_at)
VALUES
  (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())

    ");
    if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt->bind_param("isiissisi", $lead_id, $action, $old_status_id, $new_status_id, $comment, $followup_dt, $changed_by, $role, $affected_broker_id);

    break;

  case 'followup_update':
  // 🗓️ Normalize follow-up datetime (HTML5 datetime-local uses "T")
  if ($followup_dt && strpos($followup_dt, 'T') !== false) {
    $followup_dt = str_replace('T', ' ', $followup_dt);
  }

  // 🧩 Persist follow-up in broker_status table
  $sql = "INSERT INTO lead_broker_status (lead_id, broker_id, status_id, followup_datetime)
          VALUES (?, ?, 1, ?)
          ON DUPLICATE KEY UPDATE followup_datetime = VALUES(followup_datetime)";
  $fu = $conn->prepare($sql);
  if (!$fu) jsonError('Prepare failed: ' . $conn->error, 500);
  $fu->bind_param("iis", $lead_id, $broker_id, $followup_dt);
  $fu->execute();
  $fu->close();

  // 🧾 Log follow-up change in activity log
  $stmt = $conn->prepare("
  INSERT INTO lead_logs
  (lead_id, action, comment, followup_datetime, changed_by, role, affected_broker_id, created_at)
VALUES
  (?, ?, ?, ?, ?, ?, ?, NOW())

");
if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
$action = 'followup';

$stmt->bind_param("isssisi", $lead_id, $action, $comment, $followup_dt, $changed_by, $role, $affected_broker_id);


  break;


  case 'comment_only':
    // ensure row & optional follow-up
    $sql = "INSERT INTO lead_broker_status (lead_id, broker_id, status_id, followup_datetime)
            VALUES (?, ?, 1, ?)
            ON DUPLICATE KEY UPDATE followup_datetime = VALUES(followup_datetime)";
    $ensure = $conn->prepare($sql);
    if (!$ensure) jsonError('Prepare failed: ' . $conn->error, 500);
    $ensure->bind_param("iis", $lead_id, $broker_id, $followup_dt);
    $ensure->execute();
    $ensure->close();

    $stmt = $conn->prepare("
      INSERT INTO lead_logs
  (lead_id, action, comment, followup_datetime, changed_by, role, affected_broker_id, created_at)
VALUES
  (?, ?, ?, ?, ?, ?, ?, NOW())

    ");
    if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt->bind_param("isssisi", $lead_id, $action, $comment, $followup_dt, $changed_by, $role, $affected_broker_id);

    break;

    case 'reassignment':
    // Validate new broker id
    if (!$new_broker_id) jsonError('Missing new_broker_id', 400);

    // If the new broker already has the mapping, no-op
    $check = $conn->prepare("SELECT 1 FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
    if (!$check) jsonError('Prepare failed: ' . $conn->error, 500);
    $check->bind_param("ii", $lead_id, $new_broker_id);
    $check->execute();
    $check->store_result();
    if ($check->num_rows > 0) {
      $check->close();
      echo json_encode(['success' => true, 'message' => 'Broker already assigned']);
      $conn->close();
      exit;
    }
    $check->close();

    // Determine if this call is a true transfer (old broker provided AND currently mapped)
    $isTransfer = false;
    
        // If the same broker was previously unassigned for this lead, treat re-adding as a reassignment
    $wasReAdd = false;
    $chkReAdd = $conn->prepare("
      SELECT 1
      FROM lead_logs
      WHERE lead_id = ? AND action = 'unassignment' AND old_broker_id = ?
      ORDER BY id DESC
      LIMIT 1
    ");
    if (!$chkReAdd) jsonError('Prepare failed: ' . $conn->error, 500);
    $chkReAdd->bind_param("ii", $lead_id, $new_broker_id); // i=int, i=int
    $chkReAdd->execute();
    $chkReAdd->store_result();
    $wasReAdd = ($chkReAdd->num_rows > 0);
    $chkReAdd->close();

    
    if (!is_null($old_broker_id)) {
      $chkOld = $conn->prepare("SELECT 1 FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
      if (!$chkOld) jsonError('Prepare failed: ' . $conn->error, 500);
      $chkOld->bind_param("ii", $lead_id, $old_broker_id);
      $chkOld->execute();
      $chkOld->store_result();
      $isTransfer = ($chkOld->num_rows > 0);
      $chkOld->close();
    }

    // Always add the new mapping
    $stmt1 = $conn->prepare("INSERT INTO lead_broker_map (lead_id, broker_id) VALUES (?, ?)");
    if (!$stmt1) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt1->bind_param("ii", $lead_id, $new_broker_id);
    $stmt1->execute();
    $stmt1->close();

    // Reset status for the new broker
    $sql = "INSERT INTO lead_broker_status (lead_id, broker_id, status_id, followup_datetime)
            VALUES (?, ?, 1, NULL)
            ON DUPLICATE KEY UPDATE status_id = VALUES(status_id),
                                    followup_datetime = VALUES(followup_datetime)";
    $stmt2 = $conn->prepare($sql);
    if (!$stmt2) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt2->bind_param("ii", $lead_id, $new_broker_id);
    $stmt2->execute();
    $stmt2->close();

    // If this is a true transfer, remove the old mapping and status
    if ($isTransfer) {
      $delMap = $conn->prepare("DELETE FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
      if (!$delMap) jsonError('Prepare failed: ' . $conn->error, 500);
      $delMap->bind_param("ii", $lead_id, $old_broker_id);
      $delMap->execute();
      $delMap->close();

      $delStatus = $conn->prepare("DELETE FROM lead_broker_status WHERE lead_id = ? AND broker_id = ?");
      if (!$delStatus) jsonError('Prepare failed: ' . $conn->error, 500);
      $delStatus->bind_param("ii", $lead_id, $old_broker_id);
      $delStatus->execute();
      $delStatus->close();
    }

    // Validate new broker user exists
    $userStmt = $conn->prepare("SELECT id FROM users WHERE id = ?");
    if (!$userStmt) jsonError('Prepare failed: ' . $conn->error, 500);
    $userStmt->bind_param("i", $new_broker_id);
    $userStmt->execute();
    $userStmt->store_result();
    if ($userStmt->num_rows === 0) {
      $userStmt->close();
      jsonError('Invalid user ID for broker assignment', 400);
    }
    $userStmt->close();

    // 🆕 Log action: reassignment if transfer OR broker was previously unassigned. Else assignment.
if ($isTransfer || $wasReAdd) {
  $action = 'reassignment';
  $stmt = $conn->prepare("
    INSERT INTO lead_logs
      (lead_id, action, old_broker_id, new_broker_id, comment, changed_by, role, created_at)
    VALUES
      (?, ?, ?, ?, ?, ?, ?, NOW())
  ");
  if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
  // old_broker_id is NULL on re-add, non-NULL on transfer; both valid
  $stmt->bind_param(
    "isiisis",
    $lead_id, $action, $old_broker_id, $new_broker_id, $comment, $changed_by, $role
  );
} else {
  $action = 'assignment';
  $stmt = $conn->prepare("
    INSERT INTO lead_logs
      (lead_id, action, old_broker_id, new_broker_id, comment, changed_by, role, created_at)
    VALUES
      (?, ?, NULL, ?, ?, ?, ?, NOW())
  ");
  if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
  $stmt->bind_param(
    "isisss",
    $lead_id, $action, $new_broker_id, $comment, $changed_by, $role
  );
}

    break;

  case 'unassignment':
    // 🗑️ Admin removes broker from lead
    if ($role !== 'admin') jsonError('Only admins can unassign brokers', 403);
    if (!$broker_id) jsonError('Missing broker_id for unassignment', 400);

    // Log first
    $stmt = $conn->prepare("
      INSERT INTO lead_logs
        (lead_id, action, old_broker_id, new_broker_id, comment, changed_by, role, created_at)
      VALUES
        (?, ?, ?, NULL, ?, ?, ?, NOW())
    ");
    if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
    // Types: i s i s i s  → lead_id, action, old_broker_id, comment, changed_by, role
    $stmt->bind_param(
      "isisis",
      $lead_id,    // int
      $action,     // string
      $broker_id,  // int  old_broker_id
      $comment,    // string
      $changed_by, // int
      $role        // string
    );

    // Actual unassignment cleanup happens after execute (already handled below)
    break;


  default:
    jsonError('Unknown action', 400);
}

// Execute write
if ($stmt->execute()) {
  // If unassign, remove mapping + status rows for that broker
  if ($action === 'unassignment') {
    $stmt1 = $conn->prepare("DELETE FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
    if (!$stmt1) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt1->bind_param("ii", $lead_id, $broker_id);
    $stmt1->execute();
    $stmt1->close();

    $stmt2 = $conn->prepare("DELETE FROM lead_broker_status WHERE lead_id = ? AND broker_id = ?");
    if (!$stmt2) jsonError('Prepare failed: ' . $conn->error, 500);
    $stmt2->bind_param("ii", $lead_id, $broker_id);
    $stmt2->execute();
    $stmt2->close();
  }
  
  // ✅ Ensure new log record is fully flushed before response
$conn->commit(); // ensure transaction flush if autocommit off
usleep(200000); // wait 0.2s for DB write visibility to subsequent fetch


  echo json_encode(['success' => true]);
exit; // ensure no stray output is appended

} else {
  jsonError('Insert failed: ' . ($stmt->error ?: $conn->error), 500);
}

$stmt->close();
$conn->close();
