<?php
// api/leads_api.php (Knight Castle CRM)
// Returns paginated leads for Admin & Broker with filters, tab-based scope, JSON-only

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once '../config.php';
require_once '../lib/auth.php';

if (!isLoggedIn()) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Unauthorized']);
  exit;
}

$userId = (int)($_SESSION['user_id'] ?? 0);
$role   = $_SESSION['role'] ?? 'broker';

// -----------------------
// Country normalization
// -----------------------
function country_canonical_map() {
  // Add/adjust aliases here as needed; keys are lowercase aliases → canonical
  return [
    // United States
    'us' => 'United States',
    'usa' => 'United States',
    'u.s.' => 'United States',
    'u.s.a' => 'United States',
    'united states' => 'United States',
    'united states of america' => 'United States',

    // Canada
    'ca' => 'Canada',
    'can' => 'Canada',
    'canada' => 'Canada',

    // Common extras (safe defaults; keep if you want broader dedupe)
    'uk' => 'United Kingdom',
    'united kingdom' => 'United Kingdom',
    'uae' => 'United Arab Emirates',
    'united arab emirates' => 'United Arab Emirates',
    'ksa' => 'Saudi Arabia',
    'saudi arabia' => 'Saudi Arabia',
    'aus' => 'Australia',
    'australia' => 'Australia',
    'ph' => 'Philippines',
    'philippines' => 'Philippines',
    'in' => 'India',
    'india' => 'India',
  ];
}

function canonical_country_label(string $raw): string {
  $v = strtolower(trim($raw));
  if ($v === '') return '';
  $map = country_canonical_map();
  if (isset($map[$v])) return $map[$v];
  // Default: title-case the raw value
  return mb_convert_case($v, MB_CASE_TITLE, "UTF-8");
}

function variants_for_canonical(string $canonical): array {
  // Build a reverse index: canonical => [aliases..., canonical]
  $map = country_canonical_map();
  $rev = [];
  foreach ($map as $alias => $canon) {
    $rev[$canon][] = $alias; // keep alias (lowercased) for matching
  }
  // Always include the canonical itself (lowercased)
  $canonical_lc = strtolower($canonical);
  $rev[$canonical] = isset($rev[$canonical]) ? array_unique(array_merge($rev[$canonical], [$canonical_lc])) : [$canonical_lc];
  return $rev[$canonical];
}

// ✅ move action handling after we parse JSON; use JSON_EXTRACT for country
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!is_array($in)) $in = [];

if (!empty($in['action']) && $in['action'] === 'get_countries') {
  // pull distinct country values from leads.data -> $.country, trim junk, drop obvious test placeholders
  $sql = "
    SELECT DISTINCT TRIM(JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country'))) AS c
    FROM leads l
    WHERE JSON_EXTRACT(l.data, '$.country') IS NOT NULL
      AND JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country')) <> ''
      AND LOWER(JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country'))) NOT LIKE 'test%%'
      AND LOWER(JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country'))) <> 'country'
    ORDER BY c ASC
  ";
  $stmt = $conn->prepare($sql); // no binds
  $stmt->execute();
  $res  = $stmt->get_result();

  // Deduplicate by canonical label
  $canonSet = [];
  while ($row = $res->fetch_assoc()) {
    $val = $row['c'];
    if ($val === null || $val === '') continue;
    $canon = canonical_country_label($val);
    if ($canon !== '') $canonSet[$canon] = true;
  }
  $stmt->close();

  $countries = array_keys($canonSet);
  sort($countries, SORT_NATURAL | SORT_FLAG_CASE);

  echo json_encode(['ok' => true, 'countries' => $countries], JSON_UNESCAPED_UNICODE);
  exit;
}

$tab        = isset($in['tab']) ? trim((string)$in['tab']) : null;
$search     = isset($in['search']) ? trim((string)$in['search']) : '';
$searchMode = !empty($in['search_mode']);

$filters    = isset($in['filters']) && is_array($in['filters']) ? $in['filters'] : [];
$range      = isset($filters['range']) ? (string)$filters['range'] : '30days';
$statusVal  = (isset($filters['status']) && $filters['status'] !== '') ? (int)$filters['status'] : null;
$project    = isset($filters['project']) ? trim((string)$filters['project']) : '';
$platformInput = isset($filters['platform']) ? trim((string)$filters['platform']) : '';
$platformMap = [
  'Facebook' => 'fb',
  'Instagram' => 'ig',
  'Google' => 'gg',
  'Unknown' => 'unknown'
];

$platform = $platformMap[$platformInput] ?? '';

$country    = isset($filters['country']) ? trim((string)$filters['country']) : '';
$followups  = isset($filters['followups']) ? strtolower(trim((string)$filters['followups'])) : '';

$sort       = isset($in['sort']) ? strtolower((string)$in['sort']) : 'newest';
$page       = isset($in['page']) ? max(1, (int)$in['page']) : 1;
$pageSize   = isset($in['page_size']) ? max(1, min(100, (int)$in['page_size'])) : 10;
$sortDir    = ($sort === 'oldest') ? 'ASC' : 'DESC';

$selectCols = [
  "l.id",
  "l.tab_name",
  "DATE_FORMAT(l.created_at, '%Y-%m-%d %H:%i:%s') AS created_at",
  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.platform')) AS platform",
  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country')) AS country",
  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.full_name')) AS lead_name", // Correct key
  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.phone_number')) AS lead_phone", // Correct key
  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.email')) AS lead_email"
];

$from  = " FROM leads l ";
$joins = [];
$where = [];

$typesCount = '';
$paramsCount = [];
$typesData  = '';
$paramsData = [];

$brokerScoped = false;
$scopedBrokerId = null;

// === ROLE + TAB LOGIC ========================================================
if ($role === 'broker') {
  $brokerScoped = true;
  $scopedBrokerId = $userId;

  $joins[] = "INNER JOIN lead_broker_map lbm ON lbm.lead_id = l.id AND lbm.broker_id = ?";
  $typesCount .= 'i'; $paramsCount[] = $scopedBrokerId;
  $typesData  .= 'i'; $paramsData[]  = $scopedBrokerId;

  $joins[] = "LEFT JOIN lead_broker_status lbs ON lbs.lead_id = l.id AND lbs.broker_id = lbm.broker_id";
  $joins[] = "LEFT JOIN lead_statuses ls ON lbs.status_id = ls.id";

  $selectCols[] = "ls.name AS status_name";
  $selectCols[] = "lbs.followup_datetime";
} else {
  if ($role === 'admin' && $searchMode && ($tab === 'all' || $tab === null || $tab === '')) {
    // Admin is doing global search on All tab — skip scoping
  } else {
    // Everyone else — must apply scoping
    if ($tab === 'unassigned') {
      $joins[] = "LEFT JOIN lead_broker_map lbm ON lbm.lead_id = l.id";
      $where[] = "lbm.lead_id IS NULL";
    } elseif ($tab === 'all' || $tab === null || $tab === '') {
      // No scoping for admin All tab — safe
    } else {
      $maybeBrokerId = (int)$tab;
      if ($maybeBrokerId > 0) {
        $brokerScoped = true;
        $scopedBrokerId = $maybeBrokerId;

        $joins[] = "INNER JOIN lead_broker_map lbm ON lbm.lead_id = l.id AND lbm.broker_id = ?";
        $typesCount .= 'i'; $paramsCount[] = $scopedBrokerId;
        $typesData  .= 'i'; $paramsData[]  = $scopedBrokerId;

        $joins[] = "LEFT JOIN lead_broker_status lbs ON lbs.lead_id = l.id AND lbs.broker_id = lbm.broker_id";
        $joins[] = "LEFT JOIN lead_statuses ls ON lbs.status_id = ls.id";

        $selectCols[] = "ls.name AS status_name";
        $selectCols[] = "lbs.followup_datetime";
      }
    }
  }

  // Admin: show assigned brokers (when not brokerScoped)
  if (!$brokerScoped) {
    $selectCols[] = "(SELECT GROUP_CONCAT(u.name SEPARATOR ', ')
                      FROM lead_broker_map lbm2
                      JOIN users u ON u.id = lbm2.broker_id
                      WHERE lbm2.lead_id = l.id) AS assigned_brokers";
  }
}

// === FILTERS ================================================================
switch ($range) {
  case 'today':  $where[] = "DATE(l.created_at) = CURDATE()"; break;
  case '7days':  $where[] = "l.created_at >= (NOW() - INTERVAL 7 DAY)"; break;
  case '30days': $where[] = "l.created_at >= (NOW() - INTERVAL 30 DAY)"; break;
  case 'all': default: break;
}

if ($project !== '') {
  $where[] = "l.tab_name = ?";
  $typesCount .= 's'; $paramsCount[] = $project;
  $typesData  .= 's'; $paramsData[]  = $project;
}
if ($platform !== '') {
  $where[] = "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.platform')) = ?";
  $typesCount .= 's'; $paramsCount[] = $platform;
  $typesData  .= 's'; $paramsData[]  = $platform;
}

// 🔁 Country filter with alias matching (US/USA/United States, CA/Canada, etc.)
if ($country !== '') {
  $canonical = canonical_country_label($country);       // normalize UI value
  $aliases   = variants_for_canonical($canonical);      // list of lowercase aliases
  // also include the canonical label itself (lowercase) in case it’s stored as-is
  $aliases = array_values(array_unique(array_merge($aliases, [strtolower($canonical)])));

  // Build IN (?, ?, ?) with binds
  $placeholders = implode(',', array_fill(0, count($aliases), '?'));
  $where[] = "LOWER(JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.country'))) IN ($placeholders)";

  $typesCount .= str_repeat('s', count($aliases));
  $typesData  .= str_repeat('s', count($aliases));
  foreach ($aliases as $a) { $paramsCount[] = $a; $paramsData[] = $a; }
}

// 🛡️ Status & Follow-up filters
if ($brokerScoped) {
  // Status
  if ($statusVal === 0) {
    $where[] = "lbs.status_id IS NULL";
  } elseif ($statusVal === 1) {
    $where[] = "lbs.status_id = 1";
  } elseif ($statusVal !== null && $statusVal > 0) {
    $where[] = "lbs.status_id = ?";
    $typesCount .= 'i'; $paramsCount[] = $statusVal;
    $typesData  .= 'i'; $paramsData[]  = $statusVal;
  }

  // Follow-ups
  if ($followups !== '') {
    if     ($followups === 'overdue') $where[] = "lbs.followup_datetime IS NOT NULL AND lbs.followup_datetime < NOW()";
    elseif ($followups === 'today')   $where[] = "DATE(lbs.followup_datetime) = CURDATE()";
    elseif ($followups === '3d')      $where[] = "lbs.followup_datetime >= NOW() AND lbs.followup_datetime < (NOW() + INTERVAL 3 DAY)";
    elseif ($followups === '7d')      $where[] = "lbs.followup_datetime >= NOW() AND lbs.followup_datetime < (NOW() + INTERVAL 7 DAY)";
    elseif ($followups === 'gt7')     $where[] = "lbs.followup_datetime >= (NOW() + INTERVAL 7 DAY)";
  }

} else {
  if ($tab === 'unassigned') {
    // ignore status/follow-up in unassigned
  } else {
    if ($statusVal === 0) {
      $where[] = "NOT EXISTS (SELECT 1 FROM lead_broker_status s0 WHERE s0.lead_id = l.id)";
    } elseif ($statusVal === 1) {
      $where[] = "("
               . "EXISTS (SELECT 1 FROM lead_broker_status s1 WHERE s1.lead_id = l.id AND s1.status_id = 1)"
               . " OR NOT EXISTS (SELECT 1 FROM lead_broker_map mm WHERE mm.lead_id = l.id)"
               . ")";
    } elseif ($statusVal !== null && $statusVal > 0) {
      $where[] = "EXISTS (SELECT 1 FROM lead_broker_status sX WHERE sX.lead_id = l.id AND sX.status_id = ?)";
      $typesCount .= 'i'; $paramsCount[] = $statusVal;
      $typesData  .= 'i'; $paramsData[]  = $statusVal;
    }

    if ($followups !== '') {
      if     ($followups === 'overdue') $where[] = "EXISTS (SELECT 1 FROM lead_broker_status f WHERE f.lead_id = l.id AND f.followup_datetime IS NOT NULL AND f.followup_datetime < NOW())";
      elseif ($followups === 'today')   $where[] = "EXISTS (SELECT 1 FROM lead_broker_status f WHERE f.lead_id = l.id AND DATE(f.followup_datetime) = CURDATE())";
      elseif ($followups === '3d')      $where[] = "EXISTS (SELECT 1 FROM lead_broker_status f WHERE f.lead_id = l.id AND f.followup_datetime >= NOW() AND f.followup_datetime < (NOW() + INTERVAL 3 DAY))";
      elseif ($followups === '7d')      $where[] = "EXISTS (SELECT 1 FROM lead_broker_status f WHERE f.lead_id = l.id AND f.followup_datetime >= NOW() AND f.followup_datetime < (NOW() + INTERVAL 7 DAY))";
      elseif ($followups === 'gt7')     $where[] = "EXISTS (SELECT 1 FROM lead_broker_status f WHERE f.lead_id = l.id AND f.followup_datetime >= (NOW() + INTERVAL 7 DAY))";
    }
  }
}

if ($search !== '') {
  $like = '%' . $search . '%';
  $where[] = "(CAST(l.id AS CHAR) LIKE ? OR l.lead_id LIKE ? OR l.tab_name LIKE ? OR "
           .  "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.full_name')) LIKE ? OR "
           . "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.email')) LIKE ? OR "
           . "JSON_UNQUOTE(JSON_EXTRACT(l.data, '$.phone_number')) LIKE ?)";
  $typesCount .= 'ssssss'; array_push($paramsCount, $like, $like, $like, $like, $like, $like);
  $typesData  .= 'ssssss'; array_push($paramsData,  $like, $like, $like, $like, $like, $like);
}

$whereSql  = count($where) ? " WHERE " . implode(" AND ", $where) : '';
$orderSql  = " ORDER BY l.created_at $sortDir ";
$offset    = ($page - 1) * $pageSize;
$limitSql  = " LIMIT ? OFFSET ? ";
$typesData .= 'ii';
$paramsData[] = $pageSize;
$paramsData[] = $offset;

// Materialize joins and the count SQL
$joinsSql  = count($joins) ? ' ' . implode(' ', $joins) . ' ' : ' ';
$countSql  = "SELECT COUNT(DISTINCT l.id) AS total " . $from . $joinsSql . $whereSql;
$selectSql = "SELECT " . implode(', ', $selectCols) . $from . $joinsSql . $whereSql . $orderSql . $limitSql;

try {
  $stmtC = $conn->prepare($countSql);
  if ($typesCount !== '') $stmtC->bind_param($typesCount, ...$paramsCount);
  $stmtC->execute();
  $resC  = $stmtC->get_result();
  $total = (int)($resC->fetch_assoc()['total'] ?? 0);
  $stmtC->close();

  $stmt = $conn->prepare($selectSql);
  if ($typesData !== '') $stmt->bind_param($typesData, ...$paramsData);
  $stmt->execute();
  $res = $stmt->get_result();

  $rows = [];
  while ($r = $res->fetch_assoc()) {
    $assignedBrokers = isset($r['assigned_brokers']) && $r['assigned_brokers'] !== null
      ? array_map('trim', explode(',', $r['assigned_brokers']))
      : [];

    $brokerData = [];

    if ($role === 'admin' && !$brokerScoped) {
      // Fetch status/followup grouped by broker for All Leads tab
      $stmtSub = $conn->prepare("
        SELECT u.id AS broker_id, u.name AS broker_name, ls.name AS status_name, lbs.followup_datetime
        FROM lead_broker_status lbs
        LEFT JOIN lead_statuses ls ON lbs.status_id = ls.id
        JOIN users u ON lbs.broker_id = u.id
        WHERE lbs.lead_id = ?
      ");
      $stmtSub->bind_param('i', $r['id']);
      $stmtSub->execute();
      $resSub = $stmtSub->get_result();
      while ($row = $resSub->fetch_assoc()) {
        $initials = implode('', array_map(fn($part) => strtoupper($part[0]), explode(' ', $row['broker_name'])));
        $brokerData[] = [
          'broker_id'        => (int)$row['broker_id'],
          'broker_name'      => $row['broker_name'],
          'broker_initials'  => $initials,
          'status'           => $row['status_name'] ?? null,
          'followup'         => $row['followup_datetime'] ?? null,
        ];
      }
      $stmtSub->close();

      // brokerData stays as [] if no brokers assigned

    }

    $rows[] = [
      'id'                => (int)$r['id'],
      'tab_name'          => $r['tab_name'],
      'lead_name'         => $r['lead_name'] ?? null,
      'lead_phone'        => $r['lead_phone'] ?? null,
      'lead_email'        => $r['lead_email'] ?? null,
      'assigned_brokers'  => $assignedBrokers,
      'status_name'       => $r['status_name'] ?? null,
      'followup_datetime' => $r['followup_datetime'] ?? null,
      'broker_data'       => $brokerData,
      'platform'          => $r['platform'] ?? null,
      'country'           => $r['country'] ?? null,
      'created_at'        => $r['created_at'],
    ];
  }

  $stmt->close();

  echo json_encode(['ok' => true, 'total' => $total, 'data' => $rows], JSON_UNESCAPED_UNICODE);
  exit;

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Server error: ' . $e->getMessage()]);
  exit;
}
