<?php
// api/get_lead_logs.php
// Fetch activity logs for a lead (status, comments, assignments)

session_start();
require_once '../config.php';
require_once '../lib/auth.php';

header('Content-Type: application/json');
ob_clean();

function jsonError($message, $code = 500) {
  http_response_code($code);
  echo json_encode(['error' => $message]);
  exit;
}

if (!isLoggedIn()) jsonError('Unauthorized', 403);

$userId         = $_SESSION['user_id'];
$role           = $_SESSION['role'];
$leadId         = isset($_GET['lead_id']) ? (int) $_GET['lead_id'] : 0;
$brokerIdFilter = isset($_GET['broker_id']) ? (int) $_GET['broker_id'] : null;
$actionFilter   = $_GET['action'] ?? null;

if (!$leadId) jsonError('Missing lead_id', 400);

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) jsonError('DB connection failed', 500);

// Base + joins
// Base + joins
$sql = "
  SELECT
    l.*,
    s1.name AS old_status_name,
    s2.name AS new_status_name,
    COALESCE(u1.name, 'System') AS changed_by_name,
    COALESCE(u1.role, 'system') AS changed_by_role,

    -- NEW: the broker actually affected by the action
    ua.name AS affected_broker_name,

    -- Keep existing joins for assignment / unassignment context
    u2.name AS new_broker_name,
    u3.name AS old_broker_name

  FROM lead_logs l
  LEFT JOIN lead_statuses s1 ON l.old_status_id = s1.id
  LEFT JOIN lead_statuses s2 ON l.new_status_id = s2.id
  LEFT JOIN users u1 ON l.changed_by = u1.id
  LEFT JOIN users ua ON l.affected_broker_id = ua.id   -- ✅ join new column
  LEFT JOIN users u2 ON l.new_broker_id = u2.id
  LEFT JOIN users u3 ON l.old_broker_id = u3.id
  WHERE l.lead_id = ? AND l.deleted = 0
";


$params = [$leadId];
$types  = "i";

// Broker visibility — brokers only see their own affected logs
if ($role === 'broker') {
  $sql .= " AND l.affected_broker_id = ?";
  $params[] = $userId;
  $types   .= "i";
}


// Admin tab filter
if ($role === 'admin' && $brokerIdFilter) {
  // Show logs that involve that broker: affected, newly assigned, or removed
  $sql .= " AND (l.affected_broker_id = ? OR l.new_broker_id = ? OR l.old_broker_id = ?)";
  $params[] = $brokerIdFilter;
  $params[] = $brokerIdFilter;
  $params[] = $brokerIdFilter;
  $types   .= "iii";
}


// Action filter
if ($actionFilter) {
  $sql .= " AND l.action = ?";
  $params[] = $actionFilter;
  $types   .= "s";
}

$sql .= " ORDER BY l.created_at DESC";

$stmt = $conn->prepare($sql);
if (!$stmt) jsonError('Prepare failed: ' . $conn->error, 500);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$res = $stmt->get_result();

$logs = [];
while ($row = $res->fetch_assoc()) {
    $comment = trim((string)($row['comment'] ?? ''));
  if ($comment === '0') $comment = '';

  // Normalize statuses
  $oldStatus = $row['old_status_id'] === null
    ? 'Unassigned'
    : ($row['old_status_id'] == 1 ? 'New Lead' : ($row['old_status_name'] ?? null));

  $newStatus = $row['new_status_id'] === null
    ? 'Unassigned'
    : ($row['new_status_id'] == 1 ? 'New Lead' : ($row['new_status_name'] ?? null));

  // ✅ Normalize action: blank/null → 'note'
  $action = trim((string)($row['action'] ?? ''));
  // Normalize legacy alias so front-end uses one key
if ($action === 'followup_update') {
  $action = 'followup';
}

  if ($action === '') $action = 'note';
  
    // 🔁 Backfill legacy rows:
  // If action was blank in DB but a follow-up exists, treat as a FOLLOW-UP update for display.
  if (
    $action === 'note'
    && !empty($row['followup_datetime'])
    && $row['followup_datetime'] !== '0000-00-00 00:00:00'
  ) {
    $action = 'followup_update';
  }



  $logs[] = [
    'id'                => (int)$row['id'],
    'lead_id'           => (int)$row['lead_id'],
    'action'            => $action, // ✅ use normalized action
    'old_status_id'     => $row['old_status_id'],
    'new_status_id'     => $row['new_status_id'],
    'old_status_name'   => $oldStatus,
    'new_status_name'   => $newStatus,
    'comment'           => $comment,
    'followup_datetime' => $row['followup_datetime'],
    'changed_by'        => (int)$row['changed_by'],
    'role'              => $row['role'],
    'created_at'        => $row['created_at'],
    'old_broker_id'     => $row['old_broker_id'],
    'new_broker_id'     => $row['new_broker_id'],
    // ✅ Display the actually affected broker first; fallback to assignment names
'broker_name'       => $row['affected_broker_name']
                        ?: ($row['new_broker_name'] ?: ($row['old_broker_name'] ?: '—')),

    'changed_by_name'   => $row['changed_by_name'],
    'changed_by_role'   => $row['changed_by_role'],
  ];

}

echo json_encode(['logs' => $logs]);
$conn->close();
exit;
