<?php
// api/get_lead_form.php
// Returns the HTML for admin/partials/lead_form.php so the UI can refresh without a page reload.
// JSON-only. No FormData.

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/permissions.php'; // isBrokerAssignedToLead()

try {
  if (!isLoggedIn()) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
  }

  $input  = json_decode(file_get_contents('php://input'), true) ?: [];
  $leadId = isset($input['lead_id']) ? (int)$input['lead_id'] : 0;
  if ($leadId <= 0) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid lead_id']);
    exit;
  }

  $role            = $_SESSION['role']    ?? null;
  $currentBrokerId = $_SESSION['user_id'] ?? null;

  $db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
  if ($db->connect_error) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'DB connection failed']);
    exit;
  }

  // Broker ACL: only assigned brokers may view
  if ($role === 'broker') {
    if (!$currentBrokerId || !isBrokerAssignedToLead($leadId, (int)$currentBrokerId)) {
      http_response_code(403);
      echo json_encode(['success' => false, 'error' => 'Forbidden']);
      exit;
    }
  }

  // Load statuses
  $statuses = [];
  if ($res = $db->query("SELECT id, name FROM lead_statuses ORDER BY id ASC")) {
    $statuses = $res->fetch_all(MYSQLI_ASSOC);
  }

  // Load brokers for admin assign UI
  $brokers = [];
  if ($role === 'admin') {
    if ($res = $db->query("SELECT id, name FROM users WHERE role = 'broker' AND status = 'active' ORDER BY name ASC")) {
      $brokers = $res->fetch_all(MYSQLI_ASSOC);
    }
  }

  // Assigned brokers for tabs
  $assignedBrokers = [];
  $stmt = $db->prepare("
    SELECT u.id AS broker_id, u.name, lbs.status_id, lbs.followup_datetime
    FROM lead_broker_map lbm
    JOIN users u ON u.id = lbm.broker_id AND u.role = 'broker'
    LEFT JOIN lead_broker_status lbs ON lbs.lead_id = lbm.lead_id AND lbs.broker_id = lbm.broker_id
    WHERE lbm.lead_id = ?
    ORDER BY lbm.assigned_at ASC
  ");
  $stmt->bind_param("i", $leadId); // i = int
  $stmt->execute();
  $result = $stmt->get_result();
  while ($row = $result->fetch_assoc()) {
    $assignedBrokers[] = [
      'id'                => (int)$row['broker_id'],
      'name'              => $row['name'] ?? 'Unknown',
      'status_id'         => ($row['status_id'] !== null) ? (int)$row['status_id'] : 1,
      'followup_datetime' => $row['followup_datetime'] ?? ''
    ];
  }
  $stmt->close();

  // Variables expected by the partial
  $leadId         = $leadId;            // keep name as partial expects
  $role           = $role;
  $currentBrokerId= $currentBrokerId;
  // $statuses, $brokers, $assignedBrokers provided above

  ob_start();
  require __DIR__ . '/../admin/partials/lead_form.php';
  $html = ob_get_clean();

  echo json_encode(['success' => true, 'html' => $html]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success' => false, 'error' => 'Server error']);
}
