<?php
// api/dashboard_leaderboard.php
// Broker leaderboard: assigned, followups completed, qualified/booked/closed moved.

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once '../config.php';
require_once '../lib/auth.php';
if (!isLoggedIn()) { http_response_code(401); echo json_encode(['ok'=>false,'error'=>'Unauthorized']); exit; }

$role = $_SESSION['role'] ?? 'broker';
if ($role !== 'admin') { http_response_code(403); echo json_encode(['ok'=>false,'error'=>'Forbidden']); exit; }

// JSON body: { "days": 30 }
$in = json_decode(file_get_contents('php://input'), true) ?: [];
$days = max(1, min(90, (int)($in['days'] ?? 30)));

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) { http_response_code(500); echo json_encode(['ok'=>false,'error'=>'DB connection failed']); exit; }

$startExpr = "UTC_TIMESTAMP() - INTERVAL {$days} DAY";

// 1) Leads assigned per broker (lead_broker_map.assigned_at in window)
$assigned = [];
$q1 = $conn->query("
  SELECT u.id AS broker_id, u.name, COUNT(*) AS cnt
FROM lead_broker_map m
INNER JOIN users u ON u.id = m.broker_id
WHERE m.assigned_at >= ($startExpr)
  AND u.role = 'broker'

  GROUP BY u.id, u.name
");
while ($r = $q1->fetch_assoc()) $assigned[(int)$r['broker_id']] = ['name'=>$r['name'],'assigned'=>(int)$r['cnt']];

// 2) Follow-ups completed (log at/after the scheduled followup for same lead)
$completed = [];
$q2 = $conn->query("
  SELECT ll.changed_by AS broker_id, COUNT(*) AS cnt
  FROM lead_logs ll
INNER JOIN users u ON u.id = ll.changed_by AND u.role = 'broker'
INNER JOIN lead_broker_status lbs ON lbs.lead_id = ll.lead_id
  AND lbs.broker_id = ll.changed_by
  AND lbs.followup_datetime IS NOT NULL
  AND ll.created_at >= lbs.followup_datetime
WHERE (ll.deleted IS NULL OR ll.deleted=0)
  AND ll.created_at >= ($startExpr)
  AND (ll.action='status_update' OR ll.action='comment_only')
GROUP BY ll.changed_by
");
while ($r = $q2->fetch_assoc()) $completed[(int)$r['broker_id']] = (int)$r['cnt'];

// 3) Moves to Qualified / Booked / Closed
$qualified = []; $booked = []; $closed = [];
$q3 = $conn->query("
  SELECT ll.changed_by AS broker_id,
         SUM(ll.new_status_id=6)  AS q,
         SUM(ll.new_status_id=9)  AS b,
         SUM(ll.new_status_id=10) AS c
  FROM lead_logs ll
INNER JOIN users u ON u.id = ll.changed_by AND u.role = 'broker'
WHERE (ll.deleted IS NULL OR ll.deleted=0)
  AND ll.created_at >= ($startExpr)
  AND ll.new_status_id IN (6,9,10)
GROUP BY ll.changed_by
");
while ($r = $q3->fetch_assoc()) {
  $id = (int)$r['broker_id'];
  $qualified[$id] = (int)$r['q'];
  $booked[$id]    = (int)$r['b'];
  $closed[$id]    = (int)$r['c'];
}

// Collect ids from all buckets first
$ids = array_unique(array_merge(
  array_keys($assigned),
  array_keys($completed),
  array_keys($qualified),
  array_keys($booked),
  array_keys($closed)
));

// Fetch names for all those broker ids (guards against fallback labels)
$usernames = [];
if (!empty($ids)) {
  $idList = implode(',', array_map('intval', $ids));
  $qN = $conn->query("SELECT id, name FROM users WHERE id IN ($idList) AND role='broker'");
  while ($r = $qN->fetch_assoc()) {
    $usernames[(int)$r['id']] = $r['name'];
  }
}


// Merge rows (only brokers appearing in any metric)
$ids = array_unique(array_merge(array_keys($assigned), array_keys($completed), array_keys($qualified), array_keys($booked), array_keys($closed)));
$out = [];
foreach ($ids as $id) {
  $name = $usernames[$id] ?? ($assigned[$id]['name'] ?? ('Broker #'.$id));

  $out[] = [
    'broker_id' => $id,
    'broker_name' => $name,
    'assigned' => $assigned[$id]['assigned'] ?? 0,
    'followups_completed' => $completed[$id] ?? 0,
    'qualified' => $qualified[$id] ?? 0,
    'booked' => $booked[$id] ?? 0,
    'closed' => $closed[$id] ?? 0,
  ];
}

echo json_encode(['ok'=>true, 'data'=>$out], JSON_UNESCAPED_UNICODE);
