<?php
// api/dashboard_followups.php
// Returns overdue + today follow-ups. Role-aware.
// Admin → all brokers; Broker → only their own.

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once '../config.php';
require_once '../lib/auth.php';

if (!isLoggedIn()) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Unauthorized']);
  exit;
}

$userId = (int)($_SESSION['user_id'] ?? 0);
$role   = $_SESSION['role'] ?? 'broker';

// read JSON body
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true) ?: [];
$limit = max(1, min(50, (int)($in['limit'] ?? 10))); // cap at 50

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'DB connection failed']);
  exit;
}

// We want: one row per lead (earliest pending follow-up) and exclude non-actionable statuses.
// Status to exclude: "Assigned to Broker" (commonly id=2). We exclude by BOTH id and name for safety.
$excludedStatusName = 'Assigned to Broker';
$excludedStatusId   = 2;

// Common FROM/JOIN block used after we compute the earliest follow-up per lead in a subquery.
$joinBlock = "
  FROM lead_broker_status lbs
  INNER JOIN leads l           ON l.id = lbs.lead_id
  LEFT  JOIN lead_statuses ls  ON ls.id = lbs.status_id
  LEFT  JOIN users u           ON u.id = lbs.broker_id
";

$data = ['overdue' => [], 'today' => []];

/*
  Build queries:
  1) Compute MIN(followup_datetime) per lead with all filters applied.
  2) Join back to lbs on (lead_id, followup_datetime) to get the matching row’s status/broker.
*/

// Overdue
if ($role === 'admin') {
  $q1 = $conn->prepare("
    SELECT
      lbs.lead_id,
      COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.full_name')), '') AS lead_name,

      ls.name AS status_name,
      lbs.followup_datetime,
      u.name AS broker_name
    $joinBlock
    INNER JOIN (
      SELECT lead_id, MIN(followup_datetime) AS min_fu
      FROM lead_broker_status
      WHERE followup_datetime IS NOT NULL
        AND followup_datetime < NOW()
        AND (status_id <> ?)
    GROUP BY lead_id
    ) m ON m.lead_id = lbs.lead_id AND m.min_fu = lbs.followup_datetime
    WHERE (ls.name IS NULL OR ls.name <> ?)
    ORDER BY lbs.followup_datetime ASC
    LIMIT ?
  ");
  $q1->bind_param('isi', $excludedStatusId, $excludedStatusName, $limit);
} else {
  $q1 = $conn->prepare("
    SELECT
      lbs.lead_id,
      COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.full_name')), '') AS lead_name,

      ls.name AS status_name,
      lbs.followup_datetime,
      u.name AS broker_name
    $joinBlock
    INNER JOIN (
      SELECT lead_id, MIN(followup_datetime) AS min_fu
      FROM lead_broker_status
      WHERE followup_datetime IS NOT NULL
        AND followup_datetime < NOW()
        AND broker_id = ?
        AND (status_id <> ?)
      GROUP BY lead_id
    ) m ON m.lead_id = lbs.lead_id AND m.min_fu = lbs.followup_datetime
    WHERE lbs.broker_id = ?
      AND (ls.name IS NULL OR ls.name <> ?)
    ORDER BY lbs.followup_datetime ASC
    LIMIT ?
  ");
  $q1->bind_param('iiisi', $userId, $excludedStatusId, $userId, $excludedStatusName, $limit);
}

// Today
if ($role === 'admin') {
  $q2 = $conn->prepare("
    SELECT
      lbs.lead_id,
     COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.full_name')), '') AS lead_name,

      ls.name AS status_name,
      lbs.followup_datetime,
      u.name AS broker_name
    $joinBlock
    INNER JOIN (
      SELECT lead_id, MIN(followup_datetime) AS min_fu
      FROM lead_broker_status
      WHERE followup_datetime IS NOT NULL
        AND DATE(followup_datetime) = CURDATE()
        AND (status_id <> ?)
      GROUP BY lead_id
    ) m ON m.lead_id = lbs.lead_id AND m.min_fu = lbs.followup_datetime
    WHERE (ls.name IS NULL OR ls.name <> ?)
    ORDER BY lbs.followup_datetime ASC
    LIMIT ?
  ");
  $q2->bind_param('isi', $excludedStatusId, $excludedStatusName, $limit);
} else {
  $q2 = $conn->prepare("
    SELECT
      lbs.lead_id,
      COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.full_name')), '') AS lead_name,

      ls.name AS status_name,
      lbs.followup_datetime,
      u.name AS broker_name
    $joinBlock
    INNER JOIN (
      SELECT lead_id, MIN(followup_datetime) AS min_fu
      FROM lead_broker_status
      WHERE followup_datetime IS NOT NULL
        AND DATE(followup_datetime) = CURDATE()
        AND broker_id = ?
        AND (status_id <> ?)
      GROUP BY lead_id
    ) m ON m.lead_id = lbs.lead_id AND m.min_fu = lbs.followup_datetime
    WHERE lbs.broker_id = ?
      AND (ls.name IS NULL OR ls.name <> ?)
    ORDER BY lbs.followup_datetime ASC
    LIMIT ?
  ");
  $q2->bind_param('iiisi', $userId, $excludedStatusId, $userId, $excludedStatusName, $limit);
}


function broker_initials($name) {
  $parts = preg_split('/\s+/', trim($name ?: ''), -1, PREG_SPLIT_NO_EMPTY);
  if (!$parts) return '';
  $ini = '';
  foreach ($parts as $p) $ini .= mb_strtoupper(mb_substr($p, 0, 1));
  return $ini;
}

if ($q1 && $q1->execute()) {
  $r = $q1->get_result();
  while ($row = $r->fetch_assoc()) {
$data['overdue'][] = [
  'lead_id' => (int)$row['lead_id'],
  'lead_name' => $row['lead_name'] ?? '',
  'status_name' => $row['status_name'] ?? '',
  'followup_datetime' => $row['followup_datetime'],
  // always include for both roles
  'broker_name'     => $row['broker_name'] ?? '',
  'broker_initials' => broker_initials($row['broker_name'] ?? ''),
];



  }
}
if ($q2 && $q2->execute()) {
  $r = $q2->get_result();
  while ($row = $r->fetch_assoc()) {
$data['today'][] = [
  'lead_id' => (int)$row['lead_id'],
  'lead_name' => $row['lead_name'] ?? '',
  'status_name' => $row['status_name'] ?? '',
  'followup_datetime' => $row['followup_datetime'],
  'broker_name'     => $row['broker_name'] ?? '',
  'broker_initials' => broker_initials($row['broker_name'] ?? ''),
];


  }
}

echo json_encode(['ok' => true, 'data' => $data], JSON_UNESCAPED_UNICODE);
