<?php
// api/dashboard_counts.php
// JSON-only endpoint returning role-aware dashboard metrics.
// Admin: global view. Broker: scoped to own assignments.
// Uses users, leads, lead_broker_map, lead_broker_status, lead_statuses. 

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once '../config.php';
require_once '../lib/auth.php';

if (!isLoggedIn()) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Unauthorized']);
  exit;
}

$userId = (int)($_SESSION['user_id'] ?? 0);
$role   = $_SESSION['role'] ?? 'broker';

$inRaw = file_get_contents('php://input');                           // JSON only
$in    = json_decode($inRaw, true) ?: [];
$filters = is_array($in['filters'] ?? null) ? $in['filters'] : [];   // optional future filters

// Connect (match existing APIs)
$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'DB connection failed']);
  exit;
}

// Helper to fetch single INT
function fetch_count(mysqli $conn, string $sql, string $types = '', array $params = []): int {
  $stmt = $conn->prepare($sql);
  if (!$stmt) return 0;
  if ($types !== '') $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $stmt->bind_result($c);
  $c = 0;
  if ($stmt->fetch()) { /* $c set */ }
  $stmt->close();
  return (int)$c;
}

// ===== New Leads (created_at on leads table) ==============================
// Admin: all leads. Broker: only leads assigned to this broker via lead_broker_map.
$newWhere = '';
$newTypes = '';
$newParams = [];

if ($role === 'broker') {
  // Scope by assignment for brokers
  $newWhere = "INNER JOIN lead_broker_map lbm ON lbm.lead_id = l.id AND lbm.broker_id = ?";
  $newTypes = 'i';
  $newParams = [$userId];
}

$todaySql = "
  SELECT COUNT(*)
  FROM leads l
  $newWhere
  WHERE DATE(l.created_at) = CURDATE()
";
$sevenSql = "
  SELECT COUNT(*)
  FROM leads l
  $newWhere
  WHERE l.created_at >= (NOW() - INTERVAL 7 DAY)
";
$thirtySql = "
  SELECT COUNT(*)
  FROM leads l
  $newWhere
  WHERE l.created_at >= (NOW() - INTERVAL 30 DAY)
";

$newToday  = fetch_count($conn, $todaySql,  $newTypes, $newParams);  // number
$new7d     = fetch_count($conn, $sevenSql,  $newTypes, $newParams);  // number
$new30d    = fetch_count($conn, $thirtySql, $newTypes, $newParams);  // number

// ===== Unassigned Leads (admin only) =====================================
// Leads with no mapping in lead_broker_map. (Confirmed table) :contentReference[oaicite:1]{index=1}
$unassigned = 0;
if ($role === 'admin') {
  $unassigned = fetch_count(
    $conn,
    "SELECT COUNT(*) 
     FROM leads l 
     LEFT JOIN lead_broker_map m ON m.lead_id = l.id 
     WHERE m.lead_id IS NULL"
  );
}

// ===== Status distribution ===============================================
// Per-broker status stored in lead_broker_status; join lead_statuses for names. 
$statusRows = [];
if ($role === 'broker') {
  $stmt = $conn->prepare("
    SELECT s.id, s.name, COUNT(*) AS cnt
    FROM lead_broker_status lbs
    INNER JOIN lead_statuses s ON s.id = lbs.status_id
    WHERE lbs.broker_id = ? AND (lbs.deleted IS NULL OR lbs.deleted = 0)
    GROUP BY s.id, s.name
    ORDER BY s.id ASC
  ");
  $stmt->bind_param('i', $userId);
} else {
  $stmt = $conn->prepare("
    SELECT s.id, s.name, COUNT(*) AS cnt
    FROM lead_broker_status lbs
    INNER JOIN lead_statuses s ON s.id = lbs.status_id
    WHERE (lbs.deleted IS NULL OR lbs.deleted = 0)
    GROUP BY s.id, s.name
    ORDER BY s.id ASC
  ");
}
if ($stmt && $stmt->execute()) {
  $res = $stmt->get_result();
  while ($r = $res->fetch_assoc()) {
    $statusRows[] = ['id' => (int)$r['id'], 'name' => $r['name'], 'count' => (int)$r['cnt']];
  }
}
if ($stmt) $stmt->close();

// ===== Follow-up buckets (from lead_broker_status.followup_datetime) =====
// Buckets: overdue(<NOW), today, next_3_days, next_7_days (non-overlapping).
$followTypes = '';
$followParams = [];
$followWhere = " WHERE (lbs.deleted IS NULL OR lbs.deleted = 0) ";
if ($role === 'broker') {
  $followWhere .= " AND lbs.broker_id = ? ";
  $followTypes  = 'i';
  $followParams = [$userId];
}

$overdue = fetch_count(
  $conn,
  "SELECT COUNT(*) FROM lead_broker_status lbs $followWhere AND lbs.followup_datetime IS NOT NULL AND lbs.followup_datetime < NOW()",
  $followTypes, $followParams
);
$today = fetch_count(
  $conn,
  "SELECT COUNT(*) FROM lead_broker_status lbs $followWhere AND DATE(lbs.followup_datetime) = CURDATE()",
  $followTypes, $followParams
);
$next3 = fetch_count(
  $conn,
  // > today up to +3 days
  "SELECT COUNT(*) FROM lead_broker_status lbs $followWhere 
   AND lbs.followup_datetime > CURDATE()
   AND lbs.followup_datetime < (CURDATE() + INTERVAL 4 DAY)",
  $followTypes, $followParams
);
$next7 = fetch_count(
  $conn,
  // > +3 days up to +7 days
  "SELECT COUNT(*) FROM lead_broker_status lbs $followWhere 
   AND lbs.followup_datetime >= (CURDATE() + INTERVAL 4 DAY)
   AND lbs.followup_datetime <  (CURDATE() + INTERVAL 8 DAY)",
  $followTypes, $followParams
);

// ===== Platform distribution (from leads.data->$.platform) ===============
// Admin: all leads. Broker: only their assigned leads.
$platformRows = [];
if ($role === 'broker') {
  $pSql = "
    SELECT COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.platform')),'Unknown') AS platform, COUNT(*) AS cnt
    FROM leads l
    INNER JOIN lead_broker_map lbm ON lbm.lead_id = l.id AND lbm.broker_id = ?
    GROUP BY platform
    ORDER BY cnt DESC
  ";
  $pStmt = $conn->prepare($pSql);
  $pStmt->bind_param('i', $userId);
} else {
  $pSql = "
    SELECT COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.platform')),'Unknown') AS platform, COUNT(*) AS cnt
    FROM leads l
    GROUP BY platform
    ORDER BY cnt DESC
  ";
  $pStmt = $conn->prepare($pSql);
}
if ($pStmt && $pStmt->execute()) {
  $res = $pStmt->get_result();
  while ($r = $res->fetch_assoc()) {
    $platformRows[] = ['platform' => $r['platform'], 'count' => (int)$r['cnt']];
  }
}
if ($pStmt) $pStmt->close();

$conn->close();

// Final JSON response
echo json_encode([
  'ok'   => true,
  'role' => $role,
  'data' => [
    'new_leads' => ['today' => $newToday, '7d' => $new7d, '30d' => $new30d], // from leads.created_at
    'unassigned' => ($role === 'admin') ? $unassigned : null,                // admin-only metric
    'status'     => $statusRows,                                             // join lead_statuses for names
    'followups'  => ['overdue' => $overdue, 'today' => $today, 'next_3_days' => $next3, 'next_7_days' => $next7],
    'platforms'  => $platformRows
  ]
], JSON_UNESCAPED_UNICODE);
