<?php
// api/dashboard_activity.php
// Returns the latest activity items for the dashboard, role-scoped.
// Admin: last 20 across all users. Broker: last 20 where changed_by = me.

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json; charset=utf-8');

require_once '../config.php';
require_once '../lib/auth.php';

if (!isLoggedIn()) {
  http_response_code(401);
  echo json_encode(['ok' => false, 'error' => 'Unauthorized']);
  exit;
}

$userId = (int)($_SESSION['user_id'] ?? 0);
$role   = $_SESSION['role'] ?? 'broker';

// read body (JSON only)
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true) ?: [];
$limit = max(1, min(50, (int)($in['limit'] ?? 20))); // cap 50

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'DB connection failed']);
  exit;
}

// Fields: lead_logs + users.name + leads.id/name (if available)
$sqlAdmin = "
  SELECT ll.id, ll.lead_id, ll.action, ll.old_status_id, ll.new_status_id,
         ll.comment, ll.followup_datetime, ll.changed_by, ll.created_at,
         u.name AS user_name,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.name')), '')   AS lead_name,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.email')), '')  AS lead_email,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.phone')), '')  AS lead_phone,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.mobile')), '') AS lead_mobile,
         os.name AS old_status_name,
         ns.name AS new_status_name
  FROM lead_logs ll
  LEFT JOIN users u ON u.id = ll.changed_by
  LEFT JOIN leads l ON l.id = ll.lead_id
  LEFT JOIN lead_statuses os ON os.id = ll.old_status_id
  LEFT JOIN lead_statuses ns ON ns.id = ll.new_status_id
  WHERE (ll.deleted IS NULL OR ll.deleted = 0)
  ORDER BY ll.created_at DESC
  LIMIT ?
";


$sqlBroker = "
  SELECT ll.id, ll.lead_id, ll.action, ll.old_status_id, ll.new_status_id,
         ll.comment, ll.followup_datetime, ll.changed_by, ll.created_at,
         u.name AS user_name,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.name')), '')   AS lead_name,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.email')), '')  AS lead_email,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.phone')), '')  AS lead_phone,
         COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.mobile')), '') AS lead_mobile,
         os.name AS old_status_name,
         ns.name AS new_status_name
  FROM lead_logs ll
  LEFT JOIN users u ON u.id = ll.changed_by
  LEFT JOIN leads l ON l.id = ll.lead_id
  LEFT JOIN lead_statuses os ON os.id = ll.old_status_id
  LEFT JOIN lead_statuses ns ON ns.id = ll.new_status_id
  WHERE (ll.deleted IS NULL OR ll.deleted = 0)
    AND ll.changed_by = ?
  ORDER BY ll.created_at DESC
  LIMIT ?
";


if ($role === 'admin') {
  $stmt = $conn->prepare($sqlAdmin);
  if (!$stmt) { echo json_encode(['ok'=>false,'error'=>'Prepare failed']); exit; }
  $stmt->bind_param('i', $limit);                 // i = int for LIMIT
} else {
  $stmt = $conn->prepare($sqlBroker);
  if (!$stmt) { echo json_encode(['ok'=>false,'error'=>'Prepare failed']); exit; }
  $stmt->bind_param('ii', $userId, $limit);       // i = int for userId, i = int for LIMIT
}

if (!$stmt->execute()) {
  echo json_encode(['ok'=>false,'error'=>'Query failed']); exit;
}

$res = $stmt->get_result();
$out = [];
while ($r = $res->fetch_assoc()) {
  // build a safe label for the lead
$label = trim($r['lead_name'] ?? '');
if ($label === '') {
  $email = trim($r['lead_email'] ?? '');
  $phone = trim($r['lead_phone'] ?? ($r['lead_mobile'] ?? ''));
  if ($email !== '') {
    $label = preg_replace('/(.{2}).+(@.+)/', '$1****$2', $email); // mask email
  } elseif ($phone !== '') {
    $label = preg_replace('/\d(?=\d{2})/', '*', $phone);          // mask phone
  } else {
    $label = 'Lead';
  }
}

$out[] = [
  'id' => (int)$r['id'],
  'lead_id' => (int)$r['lead_id'],
  'action' => $r['action'] ?? '',
  'old_status_id' => isset($r['old_status_id']) ? (int)$r['old_status_id'] : null,
  'new_status_id' => isset($r['new_status_id']) ? (int)$r['new_status_id'] : null,
  'old_status_name' => $r['old_status_name'] ?? null,
  'new_status_name' => $r['new_status_name'] ?? null,
  'comment' => $r['comment'] ?? '',
  'followup_datetime' => $r['followup_datetime'] ?? null,
  'changed_by' => (int)$r['changed_by'],
  'created_at' => $r['created_at'],
  'user_name' => $r['user_name'] ?? 'User',
  'lead_display' => $label,                   // masked fallback
  'lead_name' => $r['lead_name'] ?? ''        // raw lead name string (may be empty)
];


}
$stmt->close();

/* Fallback: if there are no log rows, show recent lead creations.
   Admin → global. Broker → only leads assigned to them.
   Action label: 'lead_created'
*/
if (count($out) === 0) {
  if ($role === 'admin') {
    $q = $conn->prepare("
      SELECT l.id AS lead_id,
             COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.name')),'') AS lead_name,
             l.created_at
      FROM leads l
      ORDER BY l.created_at DESC
      LIMIT ?
    ");
    $q->bind_param('i', $limit);
  } else {
    $q = $conn->prepare("
      SELECT l.id AS lead_id,
             COALESCE(JSON_UNQUOTE(JSON_EXTRACT(l.data,'$.name')),'') AS lead_name,
             l.created_at
      FROM leads l
      INNER JOIN lead_broker_map lbm ON lbm.lead_id = l.id AND lbm.broker_id = ?
      ORDER BY l.created_at DESC
      LIMIT ?
    ");
    $q->bind_param('ii', $userId, $limit);
  }
  if ($q && $q->execute()) {
    $rs = $q->get_result();
    while ($r = $rs->fetch_assoc()) {
      $out[] = [
        'id' => 0,
        'lead_id' => (int)$r['lead_id'],
        'action' => 'lead_created',
        'old_status_id' => null,
        'new_status_id' => null,
        'comment' => '',
        'followup_datetime' => null,
        'changed_by' => 0,
        'created_at' => $r['created_at'],
        'user_name' => 'System',
        'lead_name' => $r['lead_name'] ?: 'Lead'
      ];
    }
  }
  if ($q) $q->close();
}

$conn->close();
echo json_encode(['ok'=>true, 'data'=>$out], JSON_UNESCAPED_UNICODE);

