<?php
// api/brokers_api.php
// Handles all broker CRUD actions (list, create/update, delete)

if (session_status() === PHP_SESSION_NONE) session_start();
require_once '../config.php';

header('Content-Type: application/json');

// 🔐 Only logged-in admin can use this API
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
  http_response_code(403);
  echo json_encode(['success' => false, 'message' => 'Unauthorized']);
  exit;
}

$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['success' => false, 'message' => 'Database connection error']);
  exit;
}

// 🧠 Determine current admin's privileges
$userId = $_SESSION['user_id'];
$isSuperAdmin = ($userId == 1);
$method = $_SERVER['REQUEST_METHOD'];

// ✅ Parse action
if ($method === 'GET') {
  $action = $_GET['action'] ?? '';
  $input = [];
} else {
  $raw = file_get_contents('php://input');
  $input = json_decode($raw, true);
  if (!is_array($input)) {
    echo json_encode(['success' => false, 'message' => 'Invalid JSON payload']);
    exit;
  }
  $action = $input['action'] ?? '';
}

switch ($action) {
  case 'fetch':
    // ✅ Load all users for table
    $stmt = $conn->prepare("SELECT id, name, company_name, email, phone, role, status FROM users ORDER BY id DESC");
    $stmt->execute();
    $result = $stmt->get_result();
    echo json_encode(['success' => true, 'data' => $result->fetch_all(MYSQLI_ASSOC)]);
    break;

  case 'delete':
    $id = (int)($input['id'] ?? 0);

    // ❌ Protect Super Admin from deletion
    if ($id === 1) {
      echo json_encode(['success' => false, 'message' => 'Super admin cannot be deleted.']);
      exit;
    }

    $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
    $stmt->bind_param('i', $id);
    $success = $stmt->execute();
    echo json_encode(['success' => $success]);
    break;

  case 'create_or_update':
    // 🔄 Sanitize + validate
    $id = (int)($input['id'] ?? 0);
    $name = trim($input['name'] ?? '');
    $company_name = trim($input['company_name'] ?? '');
    $email = trim($input['email'] ?? '');
    $phone = trim($input['phone'] ?? '');
    $role = in_array($input['role'], ['admin', 'broker']) ? $input['role'] : 'broker';
    $status = in_array($input['status'], ['active', 'inactive']) ? $input['status'] : 'active';
    $password = trim($input['password'] ?? '');

    if (!$name || !$email || !$phone || ($id === 0 && !$password)) {
      echo json_encode(['success' => false, 'message' => 'Missing required fields']);
      exit;
    }

    // 🚫 Check if email is already used
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
    $stmt->bind_param('si', $email, $id);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
      echo json_encode(['success' => false, 'message' => 'Email already in use']);
      exit;
    }

    // 🔐 Super Admin Protection
    if ($id === 1 && !$isSuperAdmin) {
      // 🚫 Only Super Admin can edit self
      echo json_encode(['success' => false, 'message' => 'Super admin can only be updated by self']);
      exit;
    }

    // 🚫 No one can change Super Admin role or demote Admins unless Super Admin
    if (!$isSuperAdmin) {
      $stmt = $conn->prepare("SELECT role FROM users WHERE id = ?");
      $stmt->bind_param('i', $id);
      $stmt->execute();
      $result = $stmt->get_result();
      $targetUser = $result->fetch_assoc();

      if ($targetUser) {
        // 🚫 Cannot downgrade Super Admin
        if ($id === 1 && $role !== 'admin') {
          echo json_encode(['success' => false, 'message' => 'Cannot change Super Admin role']);
          exit;
        }

        // 🚫 Cannot demote another Admin
        if ($targetUser['role'] === 'admin' && $role === 'broker') {
          echo json_encode(['success' => false, 'message' => 'Cannot change role of other Admins']);
          exit;
        }
      }

      // 🚫 Cannot assign 'admin' role to new user if not super admin
      if ($id === 0 && $role === 'admin') {
        echo json_encode(['success' => false, 'message' => 'Only Super Admin can assign admin role']);
        exit;
      }
    }

    // ✅ Insert or Update
    if ($id > 0) {
      // Update existing
      if ($password) {
        $hashed = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $conn->prepare("UPDATE users SET name=?, company_name=?, email=?, phone=?, role=?, status=?, password=? WHERE id=?");
        $stmt->bind_param('sssssssi', $name, $company_name, $email, $phone, $role, $status, $hashed, $id);
      } else {
        $stmt = $conn->prepare("UPDATE users SET name=?, company_name=?, email=?, phone=?, role=?, status=? WHERE id=?");
        $stmt->bind_param('ssssssi', $name, $company_name, $email, $phone, $role, $status, $id);
      }
    } else {
      // New user
      $hashed = password_hash($password, PASSWORD_DEFAULT);
      $defaultPhoto = '/assets/images/default_avatar.png';
      $stmt = $conn->prepare("INSERT INTO users (name, company_name, email, phone, role, status, password, profile_photo) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
      $stmt->bind_param('ssssssss', $name, $company_name, $email, $phone, $role, $status, $hashed, $defaultPhoto);
    }

    $success = $stmt->execute();
    echo json_encode(['success' => $success]);
    break;

  default:
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    break;
}
