<?php
// api/assign_broker.php
// Assigns a broker to a lead and logs assignment/reassignment clearly.

if (session_status() === PHP_SESSION_NONE) session_start();
header('Content-Type: application/json');

require_once '../config.php';
require_once '../lib/auth.php';

// --- Auth ---
if (!isLoggedIn() || $_SESSION['role'] !== 'admin') {
  http_response_code(403);
  echo json_encode(['error' => 'Unauthorized']);
  exit;
}

// --- Input ---
$data = json_decode(file_get_contents("php://input"), true);
$leadId   = (int)($data['lead_id'] ?? 0);
$brokerId = (int)($data['broker_id'] ?? 0);

if (!$leadId || !$brokerId) {
  http_response_code(400);
  echo json_encode(['error' => 'Missing required data']);
  exit;
}

// --- DB ---
$conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($conn->connect_error) {
  http_response_code(500);
  echo json_encode(['error' => 'DB connection failed']);
  exit;
}

// ✅ Check if this broker is already assigned to the lead
$check = $conn->prepare("SELECT 1 FROM lead_broker_map WHERE lead_id = ? AND broker_id = ?");
$check->bind_param("ii", $leadId, $brokerId);
$check->execute();
$check->store_result();

if ($check->num_rows > 0) {
  $check->close();
  echo json_encode(['success' => false, 'error' => 'Broker already assigned']);
  exit;
}
$check->close();

// ✅ Assign broker (always new mapping for that broker)
$assign = $conn->prepare("INSERT INTO lead_broker_map (lead_id, broker_id, assigned_at) VALUES (?, ?, NOW())");
$assign->bind_param("ii", $leadId, $brokerId);
$assign->execute();
$assign->close();

// ✅ Log as "assignment" (NEVER "reassignment" here)
// Each broker’s first link to a lead is an "assignment".
// True reassignments happen only via a different API action (unassign + reassign).
$adminId = $_SESSION['user_id'];
$action  = 'assignment';
$comment = "Admin assigned broker ID $brokerId to lead ID $leadId.";

$log = $conn->prepare("
  INSERT INTO lead_logs
  (lead_id, action, comment, new_broker_id, changed_by, role, created_at)
  VALUES (?, ?, ?, ?, ?, 'admin', NOW())
");
$log->bind_param("issii", $leadId, $action, $comment, $brokerId, $adminId);
$log->execute();
$log->close();

$conn->close();

echo json_encode(['success' => true]);
