<?php
// admin/update_lead.php
// Single-column compact layout. JSON-only APIs. All logic preserved.

if (session_status() === PHP_SESSION_NONE) session_start();
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/permissions.php';

if (!isLoggedIn()) {
  header("Location: admin_login.php");
  exit;
}

$role            = $_SESSION['role']    ?? null;
$currentBrokerId = $_SESSION['user_id'] ?? null;

$leadId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($leadId <= 0) {
  http_response_code(400);
  echo "<div class='p-4 text-red-400'>Invalid lead ID</div>";
  exit;
}

$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_error) {
  http_response_code(500);
  exit("DB connection failed");
}

if ($role === 'broker' && !isBrokerAssignedToLead($leadId, $currentBrokerId)) {
  http_response_code(403);
  echo "<div class='p-6 text-red-400'>❌ You’re not authorized to view this lead.</div>";
  exit;
}

// Fetch lead
$stmt = $db->prepare("SELECT * FROM leads WHERE id = ?");
$stmt->bind_param("i", $leadId);
$stmt->execute();
$lead = $stmt->get_result()->fetch_assoc();
$stmt->close();
if (!$lead) {
  http_response_code(404);
  echo "<div class='p-6 text-red-400'>❌ Lead not found.</div>";
  exit;
}

// JSON
$leadJson = json_decode($lead['data'] ?? '{}', true);

// helpers
function getFlexibleField(array $data, array $keys, $default = 'Not specified') {
  foreach ($keys as $key) if (!empty($data[$key])) return $data[$key];
  return $default;
}
function getBrokerName($db, $brokerId) {
  $stmt = $db->prepare("SELECT name FROM users WHERE id = ?");
  $stmt->bind_param("i", $brokerId);
  $stmt->execute();
  $res = $stmt->get_result()->fetch_assoc();
  $stmt->close();
  return $res['name'] ?? 'Unknown';
}

// last assigned broker
$assignedBrokerName = 'None';
$q = $db->prepare("SELECT broker_id FROM lead_broker_map WHERE lead_id = ? ORDER BY assigned_at DESC LIMIT 1");
$q->bind_param("i", $leadId);
$q->execute();
$r = $q->get_result();
if ($row = $r->fetch_assoc()) $assignedBrokerName = getBrokerName($db, (int)$row['broker_id']);
$q->close();

// broker status
$brokerStatus = [
  'status_id' => $lead['status_id'] ?? null,
  'followup_datetime' => $lead['followup_datetime'] ?? null
];
if ($role === 'broker' && $currentBrokerId) {
  $stmt = $db->prepare("SELECT status_id, followup_datetime FROM lead_broker_status WHERE lead_id = ? AND broker_id = ?");
  $stmt->bind_param("ii", $leadId, $currentBrokerId);
  $stmt->execute();
  $res = $stmt->get_result()->fetch_assoc();
  if ($res) $brokerStatus = $res;
  $stmt->close();
}

// details map
$leadDetails = [
  'id' => $lead['id'],
  'tab_name' => $lead['tab_name'] ?? '-',
  'platform' => strtoupper($leadJson['platform'] ?? '-'),
  'full_name' => $leadJson['full_name'] ?? '-',
  'phone_number' => $leadJson['phone_number'] ?? '-',
  'email' => $leadJson['email'] ?? '-',
  'budget' => $leadJson['budget'] ?? 'Not specified',
  'how_soon' => getFlexibleField($leadJson, ['how_soon', 'how_soon_are_you_planning_to_buy?']),
  'country' => $leadJson['country'] ?? '-',
  'assigned_broker' => $assignedBrokerName,
  'created_at' => $lead['created_at'] ?? '-',
  'updated_at' => $lead['updated_at'] ?? 'N/A'
];

$statusQuery = $db->query("SELECT id, name FROM lead_statuses ORDER BY id ASC");
$statuses = $statusQuery ? $statusQuery->fetch_all(MYSQLI_ASSOC) : [];

$brokerQuery = $db->query("SELECT id, name FROM users WHERE role='broker' AND status='active' ORDER BY name ASC");
$brokers = $brokerQuery ? $brokerQuery->fetch_all(MYSQLI_ASSOC) : [];

$assignedBrokers = [];
$stmt = $db->prepare("
  SELECT u.id AS broker_id, u.name, lbs.status_id, lbs.followup_datetime
  FROM lead_broker_map lbm
  JOIN users u ON u.id = lbm.broker_id AND u.role = 'broker'
  LEFT JOIN lead_broker_status lbs ON lbs.lead_id = lbm.lead_id AND lbs.broker_id = lbm.broker_id
  WHERE lbm.lead_id = ?
  ORDER BY lbm.assigned_at ASC
");
$stmt->bind_param("i", $leadId);
$stmt->execute();
$res = $stmt->get_result();
while ($row = $res->fetch_assoc()) {
  $assignedBrokers[] = [
    'id' => (int)$row['broker_id'],
    'name' => $row['name'] ?? 'Unknown',
    'status_id' => ($row['status_id'] !== null) ? (int)$row['status_id'] : 1,
    'followup_datetime' => $row['followup_datetime'] ?? ''
  ];
}
$stmt->close();

$pageTitle = "Update Lead";
ob_start();
?>

<!-- Wrapper -->
<div class="space-y-6 text-sm leading-relaxed">

  <!-- Compact summary -->
  <section class="bg-gray-800/80 border border-gray-700 rounded-lg p-3 sm:p-4">
    <div class="flex flex-wrap items-center gap-2 sm:gap-3">
      <div class="flex items-center gap-2">
        <span class="inline-flex items-center justify-center w-7 h-7 rounded-full bg-blue-600 text-white text-xs font-semibold">
          <?= strtoupper(substr($leadDetails['full_name'] ?: 'U', 0, 1)) ?>
        </span>
        <div>
          <div class="text-white font-medium text-[13px]"><?= htmlspecialchars($leadDetails['full_name']) ?></div>
          <div class="text-[12px] text-gray-400">
            <?= htmlspecialchars($leadDetails['email']) ?> · <?= htmlspecialchars($leadDetails['phone_number']) ?>
          </div>
        </div>
      </div>
      <span class="text-gray-600">•</span>
      <div class="text-[13px] text-gray-300">
        <?= htmlspecialchars($leadDetails['tab_name']) ?> · <?= htmlspecialchars($leadDetails['platform']) ?>
      </div>
      <span class="text-gray-600 hidden sm:inline">•</span>
      <div class="text-[13px] text-gray-300">
        <i class="fas fa-user-tie mr-1 text-gray-400"></i>
        <?= htmlspecialchars($leadDetails['assigned_broker']) ?>
      </div>
    </div>
  </section>

  <!-- Assign (admin) -->
  <?php
  if ($role === 'admin') {
    $assignedIds = array_map(fn($b) => (int)$b['id'], $assignedBrokers);
    $unassigned = array_filter($brokers, fn($b) => !in_array((int)$b['id'], $assignedIds, true));
    echo '<div id="assignContainer">';
    if (!empty($unassigned)) {
      echo '<form id="assignBrokerForm" class="flex flex-wrap items-center gap-2 bg-gray-800/70 p-3 rounded border border-gray-700" data-lead-id="' . $leadId . '">';
      echo '  <span class="text-gray-300 text-[13px]"><i class="fas fa-user-plus mr-1"></i>Assign Broker</span>';
      echo '  <select id="newBrokerSelect" class="bg-gray-800 border border-gray-600 rounded text-sm p-1.5 text-white">';
      echo '    <option value="" disabled selected>➕ Select broker...</option>';
      foreach ($unassigned as $ub) {
        echo '<option value="' . htmlspecialchars($ub['id']) . '">' . htmlspecialchars($ub['name']) . '</option>';
      }
      echo '  </select>';
      echo '  <button type="button" id="assignBrokerBtn" class="bg-blue-600 hover:bg-blue-700 text-white text-xs font-medium rounded px-3 py-1.5">Assign</button>';
      echo '</form>';
    }
    echo '</div>';
  }
  ?>

  <!-- Lead Details -->
  <section class="bg-gray-800/60 border border-gray-700 rounded-lg p-3 sm:p-4">
    <?php require __DIR__ . '/partials/lead_details.php'; ?>
  </section>

  <!-- Lead Form -->
  <section id="leadFormContainer" class="bg-gray-800/60 border border-gray-700 rounded-lg p-3 sm:p-4">
    <?php require __DIR__ . '/partials/lead_form.php'; ?>
  </section>

  <!-- Timeline -->
  <section class="bg-gray-800/60 border border-gray-700 rounded-lg p-3 sm:p-4 mb-10">
    <?php require __DIR__ . '/partials/lead_logs.php'; ?>
  </section>
</div>

<script src="/assets/js/update_lead.js"></script>

<?php
$mainContent = ob_get_clean();
include __DIR__ . '/base_layout.php';
?>
