<?php
// admin/lead_logs.php — Timeline wrapper for logs (per broker tab view, role filtered)

if (session_status() === PHP_SESSION_NONE) session_start();
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/permissions.php';

if (!isLoggedIn()) {
  http_response_code(403);
  exit("Unauthorized");
}

$leadId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if (!$leadId) {
  echo "<div class='p-6 text-red-400'>Invalid lead ID.</div>";
  exit;
}

$role   = $_SESSION['role'];
$userId = $_SESSION['user_id'];

$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);

// ✅ NEW: Fetch brokers assigned to this lead using lead_broker_map
$brokerQuery = $db->prepare("
  SELECT DISTINCT u.id AS broker_id, u.name 
  FROM lead_broker_map lbm
  JOIN users u ON lbm.broker_id = u.id
  WHERE lbm.lead_id = ? AND u.role = 'broker'
  ORDER BY u.name
");
$brokerQuery->bind_param("i", $leadId);
$brokerQuery->execute();
$brokers = $brokerQuery->get_result()->fetch_all(MYSQLI_ASSOC);
$brokerQuery->close();

// 🔒 Show only assigned brokers; restrict tab visibility based on role
$visibleBrokers = [];
foreach ($brokers as $b) {
  if ($role === 'admin' || $b['broker_id'] == $userId) {
    $visibleBrokers[] = $b;
  }
}
// 🔎 DEBUG: Output list of visible brokers to check tab generation logic
echo '<pre style="color:lime">';
print_r($visibleBrokers);
echo '</pre>';
?>

<div class="bg-surface p-4 rounded shadow">
  <h2 class="text-lg font-semibold mb-4 text-gray-200">🕓 Lead Activity Timeline</h2>

  <!-- 🔘 Broker Filter Tabs -->
  <div class="flex flex-wrap gap-2 mb-4">
    <?php if ($role === 'admin'): ?>
      <button class="logTabBtn px-3 py-1 rounded bg-blue-700 text-white text-sm" data-broker="">All Activity</button>
    <?php endif; ?>

    <?php foreach ($visibleBrokers as $b): ?>
      <button class="logTabBtn px-3 py-1 rounded bg-gray-700 text-gray-300 text-sm"
              data-broker="<?= $b['broker_id'] ?>">
        <?= htmlspecialchars($b['name']) ?>
      </button>
    <?php endforeach; ?>
  </div>

  <!-- 🔽 Optional Filter by Action Type -->
  <div class="mb-4">
    <label for="actionFilter" class="text-sm text-gray-300 mr-2">Filter by:</label>
    <select id="actionFilter" class="bg-gray-800 text-white text-sm rounded px-2 py-1">
      <option value="all">All</option>
      <option value="status_update">Status Updates</option>
      <option value="comment_only">Comments</option>
      <option value="reassignment">Reassignments</option>
      <option value="unassignment">Unassignments</option>
    </select>
  </div>

  <!-- 📜 Logs Container -->
  <div id="leadLogsContainer" class="space-y-3 text-sm text-gray-200">
    <!-- JS will inject logs here -->
  </div>
</div>

<script>
// ✅ Timeline JS: Triggered on tab switch or filter change
let currentBrokerId = '';
let currentFilter = 'all';

// 🔘 Handle tab clicks
document.querySelectorAll('.logTabBtn').forEach(btn => {
  btn.addEventListener('click', () => {
    document.querySelectorAll('.logTabBtn').forEach(b => b.classList.remove('bg-blue-700', 'text-white'));
    btn.classList.add('bg-blue-700', 'text-white');

    currentBrokerId = btn.getAttribute('data-broker') || '';
    fetchLogs(currentBrokerId, currentFilter);
  });
});

// 🔄 Handle filter change
document.getElementById('actionFilter').addEventListener('change', e => {
  currentFilter = e.target.value;
  fetchLogs(currentBrokerId, currentFilter);
});

// 🚀 Initial fetch — show all activity (admin) or personal (broker)
fetchLogs('', 'all');

// 🔁 Fetch logs from API and render into DOM
function fetchLogs(brokerId, filter) {
  const leadId = <?= $leadId ?>;
  const url = `/api/get_lead_logs.php?lead_id=${leadId}&broker_id=${brokerId}&filter=${filter}`;

  fetch(url)
    .then(res => res.json())
    .then(data => {
      const logs = (data.logs || []); // 🛑 .reverse() removed — server already returns DESC
      const container = document.getElementById('leadLogsContainer');
      container.innerHTML = '';

      if (!logs.length) {
        container.innerHTML = '<p class="text-gray-400">No logs found.</p>';
        return;
      }

      logs.forEach(log => {
        const wrapper = document.createElement('div');
        wrapper.className = 'bg-gray-800 rounded p-3';

        // 🎯 Label by action
        let badge = { label: '', color: '' };
        switch (log.action) {
  case 'status_update': badge = { label: 'STATUS UPDATE', color: 'bg-blue-600' }; break;
  case 'comment_only':  badge = { label: 'COMMENT',       color: 'bg-yellow-600' }; break;
  case 'reassignment':  badge = { label: 'REASSIGNED',    color: 'bg-purple-600' }; break;
  case 'unassignment':  badge = { label: 'UNASSIGNED',    color: 'bg-red-600' }; break;
 case 'assignment':    badge = { label: 'ASSIGNED',      color: 'bg-green-700' }; break;
 case 'followup':      badge = { label: 'FOLLOW-UP',     color: 'bg-teal-600' }; break;
  default:              badge = { label: 'LOG',           color: 'bg-gray-600' };
}

        // 🕓 Timestamp formatting
        const logTime = new Date(log.created_at);
        const timestamp = logTime.toLocaleString();

        // 🔁 Status change (if any)
        const statusChange = (log.action === 'status_update' && log.old_status_id && log.new_status_id && log.old_status_id !== log.new_status_id)
          ? `<div class="text-green-400 text-xs italic mt-1">Status: ${log.old_status_id} → ${log.new_status_id}</div>` : '';

        // 🟡 Follow-up block (if exists)
        let followupBlock = '';
        if (log.followup_datetime && log.followup_datetime !== '0000-00-00 00:00:00') {
          const followupDate = new Date(log.followup_datetime);
          const isOverdue = followupDate < new Date();
          followupBlock = `
            <div class="flex items-center gap-2 mt-1">
              <span class="text-xs text-gray-300">Follow-up: ${followupDate.toLocaleString()}</span>
              ${isOverdue ? '<span class="text-xs text-red-400 font-semibold bg-red-700 px-2 py-0.5 rounded">OVERDUE</span>' : ''}
              <span class="text-xs text-yellow-400 font-semibold bg-yellow-700 px-2 py-0.5 rounded">FOLLOW-UP</span>
            </div>
          `;
        }

        // 🧾 Final render
        wrapper.innerHTML = `
          <div class="flex items-center gap-2 mb-1">
            <span class="px-2 py-0.5 rounded text-xs font-bold uppercase ${badge.color} text-white">${badge.label}</span>
            <span class="text-xs text-gray-400 ml-auto">${timestamp}</span>
          </div>
          <div class="text-sm text-white">${log.comment || '-'}</div>
          ${statusChange}
          ${followupBlock}
          <div class="text-xs text-gray-400 mt-2 italic">
            Affected Broker: ${log.target_broker_name || '—'}<br>
            By ${log.changed_by_name || 'Deleted User'} (${log.role || ''})
          </div>
        `;

        container.appendChild(wrapper);
      });
    })
    .catch(err => {
      console.error('Log fetch error:', err);
      document.getElementById('leadLogsContainer').innerHTML = '<p class="text-red-500">Error loading logs.</p>';
    });
}
</script>
