<?php
session_start();
require_once '../config.php';
date_default_timezone_set('Asia/Dubai');

if (!isset($_SESSION["user_id"]) || $_SESSION["role"] !== 'admin') {
    header("Location: admin_login.php");
    exit;
}

// 🔍 Filter Values
$statuses = $conn->query("SELECT * FROM lead_statuses ORDER BY id")->fetch_all(MYSQLI_ASSOC);
$projectTabs = $conn->query("SELECT DISTINCT tab_name FROM leads ORDER BY tab_name")->fetch_all(MYSQLI_ASSOC);
$countries = $conn->query("SELECT DISTINCT JSON_UNQUOTE(JSON_EXTRACT(data, '$.country')) as country FROM leads WHERE JSON_EXTRACT(data, '$.country') IS NOT NULL")->fetch_all(MYSQLI_ASSOC);
$platforms = $conn->query("SELECT DISTINCT JSON_UNQUOTE(JSON_EXTRACT(data, '$.platform')) as platform FROM leads WHERE JSON_EXTRACT(data, '$.platform') IS NOT NULL")->fetch_all(MYSQLI_ASSOC);
$brokers = $conn->query("SELECT id, name FROM brokers WHERE role='broker' ORDER BY name")->fetch_all(MYSQLI_ASSOC);

// 🧠 Filters
$where = [];
$params = [];
$types = '';

if (!empty($_GET['status_id'])) {
    $where[] = "leads.status_id = ?";
    $params[] = $_GET['status_id'];
    $types .= 'i';
}
if (!empty($_GET['broker_id'])) {
    $where[] = "leads.broker_id = ?";
    $params[] = $_GET['broker_id'];
    $types .= 'i';
}
if (!empty($_GET['start_date'])) {
    $where[] = "leads.created_at >= ?";
    $params[] = $_GET['start_date'] . " 00:00:00";
    $types .= 's';
}
if (!empty($_GET['end_date'])) {
    $where[] = "leads.created_at <= ?";
    $params[] = $_GET['end_date'] . " 23:59:59";
    $types .= 's';
}
if (!empty($_GET['tab_name'])) {
    $where[] = "leads.tab_name = ?";
    $params[] = $_GET['tab_name'];
    $types .= 's';
}
if (!empty($_GET['country'])) {
    $where[] = "JSON_UNQUOTE(JSON_EXTRACT(data, '$.country')) = ?";
    $params[] = $_GET['country'];
    $types .= 's';
}
if (!empty($_GET['platform'])) {
    $where[] = "JSON_UNQUOTE(JSON_EXTRACT(data, '$.platform')) = ?";
    $params[] = $_GET['platform'];
    $types .= 's';
}
if (!empty($_GET['search'])) {
    $where[] = "(LOWER(data) LIKE ?)";
    $params[] = "%" . strtolower($_GET['search']) . "%";
    $types .= 's';
}

// 📄 Pagination
$perPage = 10;
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

// 🔢 Count total for pagination
$countSql = "SELECT COUNT(*) as total FROM leads";
if (!empty($where)) $countSql .= " WHERE " . implode(" AND ", $where);
$countStmt = $conn->prepare($countSql);
if (!empty($params)) $countStmt->bind_param($types, ...$params);
$countStmt->execute();
$totalLeads = $countStmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalLeads / $perPage);

// 🧾 Final query
$sql = "SELECT leads.*, brokers.name AS broker_name FROM leads
        LEFT JOIN brokers ON leads.broker_id = brokers.id";
if (!empty($where)) $sql .= " WHERE " . implode(" AND ", $where);
// 🔐 Whitelist allowed sort columns to prevent SQL injection
$allowedSortBy = [
    'id' => 'leads.id',
    'created_at' => 'leads.created_at',
    'tab_name' => 'leads.tab_name',
    'full_name' => "JSON_UNQUOTE(JSON_EXTRACT(leads.data, '$.full_name'))",
    'platform' => "JSON_UNQUOTE(JSON_EXTRACT(leads.data, '$.platform'))",
    'country' => "JSON_UNQUOTE(JSON_EXTRACT(leads.data, '$.country'))",
    'status_id' => 'leads.status_id',
    'broker_name' => 'broker_name' // ✅ New
];


// 🧠 Determine the sort column from query or fallback to created_at
$sortByKey = $_GET['sort_by'] ?? 'created_at';
$sortByColumn = $allowedSortBy[$sortByKey] ?? 'leads.created_at';

// 🧭 Determine sort direction (asc/desc), default to DESC
$sortDir = strtolower($_GET['sort'] ?? 'desc') === 'asc' ? 'ASC' : 'DESC';

// 🧾 Apply sorting to final query
// ✨ Apply secondary sorting if not already sorting by created_at
if ($sortByColumn !== 'leads.created_at') {
    $sql .= " ORDER BY $sortByColumn $sortDir, leads.created_at DESC";
} else {
    $sql .= " ORDER BY $sortByColumn $sortDir";
}
$sql .= " LIMIT $perPage OFFSET $offset";



$stmt = $conn->prepare($sql);
if (!empty($params)) $stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
?>

<!DOCTYPE html>
<html>
<head>
    <title>Admin Dashboard</title>
    <style>
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 8px; border: 1px solid #ccc; text-align: left; }
        th { background: #f2f2f2; }
        form.inline-form { display: inline; }
        .pagination a { margin-right: 10px; }
    </style>
</head>
<body>

<h1>Welcome, <?= htmlspecialchars($_SESSION["name"] ?? 'Admin') ?></h1>
<a href="logout.php">Logout</a>

<!-- 🔎 Filter Form -->
<form method="GET">
    <div style="display: flex; flex-wrap: wrap; gap: 10px; margin-top: 20px;">
        <select name="status_id"><option value="">All Statuses</option>
            <?php foreach ($statuses as $s): ?>
                <option value="<?= $s['id'] ?>" <?= ($_GET['status_id'] ?? '') == $s['id'] ? 'selected' : '' ?>>
                    <?= htmlspecialchars($s['name']) ?>
                </option>
            <?php endforeach; ?>
        </select>

        <select name="broker_id"><option value="">All Brokers</option>
            <?php foreach ($brokers as $b): ?>
                <option value="<?= $b['id'] ?>" <?= ($_GET['broker_id'] ?? '') == $b['id'] ? 'selected' : '' ?>>
                    <?= htmlspecialchars($b['name']) ?>
                </option>
            <?php endforeach; ?>
        </select>

        <input type="date" name="start_date" value="<?= $_GET['start_date'] ?? '' ?>">
        <input type="date" name="end_date" value="<?= $_GET['end_date'] ?? '' ?>">

        <select name="tab_name"><option value="">All Projects</option>
            <?php foreach ($projectTabs as $t): ?>
                <option value="<?= $t['tab_name'] ?>" <?= ($_GET['tab_name'] ?? '') == $t['tab_name'] ? 'selected' : '' ?>>
                    <?= htmlspecialchars($t['tab_name']) ?>
                </option>
            <?php endforeach; ?>
        </select>

        <select name="country"><option value="">All Countries</option>
            <?php foreach ($countries as $c): ?>
                <option value="<?= $c['country'] ?>" <?= ($_GET['country'] ?? '') == $c['country'] ? 'selected' : '' ?>>
                    <?= htmlspecialchars($c['country']) ?>
                </option>
            <?php endforeach; ?>
        </select>

        <select name="platform"><option value="">All Platforms</option>
            <?php foreach ($platforms as $p): ?>
                <option value="<?= $p['platform'] ?>" <?= ($_GET['platform'] ?? '') == $p['platform'] ? 'selected' : '' ?>>
                    <?= htmlspecialchars($p['platform']) ?>
                </option>
            <?php endforeach; ?>
        </select>

        <input type="text" name="search" placeholder="Search..." value="<?= $_GET['search'] ?? '' ?>">
                <!-- 📊 Sorting Column -->
        <select name="sort_by">
            <?php
            // Map of user-facing labels to actual DB columns
            $sortOptions = [
                'id' => 'ID',
                'created_at' => 'Created At',
                'tab_name' => 'Project',
                'full_name' => 'Full Name',
                'platform' => 'Platform',
                'country' => 'Country',
                'status_id' => 'Status',
                'broker_name' => 'Broker',
            ];
            foreach ($sortOptions as $key => $label): ?>
                <option value="<?= $key ?>" <?= ($_GET['sort_by'] ?? 'created_at') == $key ? 'selected' : '' ?>>
                    Sort by <?= $label ?>
                </option>
            <?php endforeach; ?>
        </select>

        <!-- ⬆️⬇️ Sort Direction -->
        <select name="sort">
            <option value="desc" <?= ($_GET['sort'] ?? 'desc') === 'desc' ? 'selected' : '' ?>>Newest First</option>
            <option value="asc" <?= ($_GET['sort'] ?? '') === 'asc' ? 'selected' : '' ?>>Oldest First</option>
        </select>

        <button type="submit">Filter</button>
        <a href="admin_dashboard.php">Reset</a>
    </div>
</form>

<!-- 📊 Data Table -->
<h2>All Leads</h2>
<table>
    <thead>
        <tr>
            <th>ID</th>
            <th>Created At</th>
            <th>Project</th>
            <th>Full Name</th>
            <th>Email</th>
            <th>Phone</th>
            <th>Platform</th>
            <th>Country</th>
            <th>Broker</th>
            <th>Status</th>
            <th>Assign / Update</th>
        </tr>
    </thead>
    <tbody>
    <?php while ($row = $result->fetch_assoc()):
        $data = json_decode($row['data'], true);
        $platformMap = ['fb' => 'Facebook', 'ig' => 'Instagram', 'gg' => 'Google'];
        $platform = $platformMap[$data['platform'] ?? ''] ?? ($data['platform'] ?? '');
    ?>
        <tr>
            <td><?= $row['id'] ?></td>
            <td><?= htmlspecialchars($row['created_at']) ?></td>
            <td><?= htmlspecialchars($row['tab_name'] ?? '') ?></td>
            <td><?= htmlspecialchars($data['full_name'] ?? '') ?></td>
            <td><?= htmlspecialchars($data['email'] ?? '') ?></td>
            <td><?= htmlspecialchars($data['phone_number'] ?? '') ?></td>
            <td><?= htmlspecialchars($platform) ?></td>
            <td><?= htmlspecialchars($data['country'] ?? '') ?></td>
            <td><?= htmlspecialchars($row['broker_name'] ?? 'Unassigned') ?></td>
            <td>
                <?php foreach ($statuses as $s): if ($s['id'] == $row['status_id']) echo htmlspecialchars($s['name']); endforeach; ?>
            </td>
            <td>
                <form method="POST" action="update_lead.php" class="inline-form">
                    <input type="hidden" name="lead_id" value="<?= $row['id'] ?>">
                    <select name="broker_id">
                        <option value="">--Assign--</option>
                        <?php foreach ($brokers as $b): ?>
                            <option value="<?= $b['id'] ?>" <?= $row['broker_id'] == $b['id'] ? 'selected' : '' ?>>
                                <?= $b['name'] ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <select name="status_id">
                        <?php foreach ($statuses as $s): ?>
                            <option value="<?= $s['id'] ?>" <?= $row['status_id'] == $s['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($s['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit">Update</button>
                </form>
            </td>
        </tr>
    <?php endwhile; ?>
    </tbody>
</table>

<!-- 📄 Pagination -->
<div class="pagination" style="margin-top: 20px;">
    <strong>Pages:</strong>
    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <?php $query = $_GET; $query['page'] = $i; $link = '?' . http_build_query($query); ?>
        <a href="<?= $link ?>" <?= $i == $page ? 'style="font-weight:bold;"' : '' ?>><?= $i ?></a>
    <?php endfor; ?>
</div>

</body>
</html>